<!-- Copyright 2011 The MathWorks, Inc.

     This file converts a project in the new MATLAB Coder format from R2012a to the format used in prior releases.
     It is being used to actually save in the old format in order to maintain temporary forward compatibility
     and compatibility with non-project-infrastructure code that depends on the old format.
-->

<xsl:stylesheet version="1.0" xmlns:xsl="http://www.w3.org/1999/XSL/Transform">
    <xsl:output method="xml" indent="yes" />

    <xsl:template match="configuration">
        <configuration>

            <!-- In earlier releases, we had two possible targets for a MATLAB Coder project: one with Embedded Coder
                 support and one without. Now there is only one target, but when downgrading a project back to the
                 older format, we need to restore the original target. -->
            <xsl:choose>
                <xsl:when test="param.HasECoderFeatures = 'true'">
                    <xsl:attribute name="target">target.matlab.ecoder</xsl:attribute>
                </xsl:when>
                <xsl:otherwise>
                    <xsl:attribute name="target">target.matlab.coder</xsl:attribute>
                </xsl:otherwise>
            </xsl:choose>

            <xsl:apply-templates select="*" />

            <!-- The format of the hardware section changes entirely -->
            <param.hardware.implementation>
                <HardwareImplementation>
                    <ProdEqTarget><xsl:value-of select="profile[@key = 'profile.c']/param.SameHardware" /></ProdEqTarget>
                    <xsl:call-template name="hardware">
                        <xsl:with-param name="oldPrefix" select="'Target'" />
                        <xsl:with-param name="newSuffix" select="'Target'" />
                    </xsl:call-template>

                    <xsl:call-template name="hardware">
                        <xsl:with-param name="oldPrefix" select="'Prod'" />
                        <xsl:with-param name="newSuffix" select="'Production'" />
                    </xsl:call-template>
                </HardwareImplementation>
            </param.hardware.implementation>

            <!-- Merge the unset sections -->
            <unset>
                <xsl:apply-templates select="descendant::unset/*" />
            </unset>
        </configuration>
    </xsl:template>

    <!-- Drop the plugin-version so we don't make the infrastructure think the output is in the new format -->
    <xsl:template match="@plugin-version" />

    <!-- Filter out the individual hardware settings, since they are special cased by the configuration template above. -->
    <xsl:template match="*[contains(name(), '.Hardware')]" />

    <!-- Filter out the unset sections since those are also special-cased -->
    <xsl:template match="unset" />

    <!-- Filter out profile root elements but not the content -->
    <xsl:template match="profile">
        <xsl:apply-templates select="*"/>
    </xsl:template>

    <xsl:template match="*[contains(name(), '.TargetLang')]">
        <xsl:call-template name="copyWithPrefix"><xsl:with-param name="extraPrefix" select="'.general'" /></xsl:call-template>
    </xsl:template>

    <xsl:template match="*[contains(name(), '.IntegrityChecks')]">
        <xsl:call-template name="copyWithPrefix"><xsl:with-param name="extraPrefix" select="'.general'" /></xsl:call-template>
    </xsl:template>

    <xsl:template match="*[contains(name(), '.ResponsivenessChecks')]">
        <xsl:call-template name="copyWithPrefix"><xsl:with-param name="extraPrefix" select="'.general'" /></xsl:call-template>
    </xsl:template>

    <xsl:template match="*[contains(name(), '.EnableBLAS')]">
        <xsl:call-template name="copyWithPrefix"><xsl:with-param name="extraPrefix" select="'.general'" /></xsl:call-template>
    </xsl:template>

    <xsl:template match="*[contains(name(), '.ExtrinsicCalls')]">
        <xsl:call-template name="copyWithPrefix"><xsl:with-param name="extraPrefix" select="'.general'" /></xsl:call-template>
    </xsl:template>

    <xsl:template match="*[contains(name(), '.EchoExpressions')]">
        <xsl:call-template name="copyWithPrefix"><xsl:with-param name="extraPrefix" select="'.general'" /></xsl:call-template>
    </xsl:template>

    <xsl:template match="*[contains(name(), '.EnableDebugging')]">
        <xsl:call-template name="copyWithPrefix"><xsl:with-param name="extraPrefix" select="'.general'" /></xsl:call-template>
    </xsl:template>

    <xsl:template match="*[contains(name(), '.Verbose')]">
        <xsl:call-template name="copyWithPrefix" />
    </xsl:template>

    <xsl:template match="*[contains(name(), '.SaturateOnIntegerOverflow')]">
        <xsl:call-template name="copyWithPrefix"><xsl:with-param name="extraPrefix" select="'.general'" /></xsl:call-template>
    </xsl:template>

    <xsl:template match="*[contains(name(), '.FilePartitionMethod')]">
        <xsl:call-template name="copyWithPrefix"><xsl:with-param name="extraPrefix" select="'.general'" /></xsl:call-template>
    </xsl:template>

    <xsl:template match="*[contains(name(), '.GlobalDataSyncMethod')]">
        <xsl:call-template name="copyWithPrefix"><xsl:with-param name="extraPrefix" select="'.general'" /></xsl:call-template>
    </xsl:template>

    <xsl:template match="*[contains(name(), '.EnableVariableSizing')]">
        <xsl:call-template name="copyWithPrefix"><xsl:with-param name="extraPrefix" select="'.general'" /></xsl:call-template>
    </xsl:template>

    <xsl:template match="*[contains(name(), '.DynamicMemoryAllocation')]">
        <xsl:call-template name="copyWithPrefix"><xsl:with-param name="extraPrefix" select="'.general'" /></xsl:call-template>
    </xsl:template>

    <xsl:template match="*[contains(name(), '.GenerateReport')]">
        <xsl:call-template name="copyWithPrefix"><xsl:with-param name="extraPrefix" select="'.report'" /></xsl:call-template>
    </xsl:template>

    <xsl:template match="*[contains(name(), '.LaunchReport')]">
        <xsl:call-template name="copyWithPrefix"><xsl:with-param name="extraPrefix" select="'.report'" /></xsl:call-template>
    </xsl:template>

    <xsl:template match="*[contains(name(), '.GenerateComments')]">
        <xsl:call-template name="copyWithPrefix"><xsl:with-param name="extraPrefix" select="'.comments'" /></xsl:call-template>
    </xsl:template>

    <xsl:template match="*[contains(name(), '.MATLABSourceComments')]">
        <xsl:call-template name="copyWithPrefix"><xsl:with-param name="extraPrefix" select="'.comments'" /></xsl:call-template>
    </xsl:template>

    <xsl:template match="*[contains(name(), '.MATLABFcnDesc')]">
        <xsl:call-template name="copyWithPrefix"><xsl:with-param name="cProfile" select="'ert'" /></xsl:call-template>
    </xsl:template>

    <xsl:template match="*[contains(name(), '.CustomSymbolStrGlobalVar')]">
        <xsl:call-template name="copyWithPrefix"><xsl:with-param name="cProfile" select="'ert'" /></xsl:call-template>
    </xsl:template>

    <xsl:template match="*[contains(name(), '.CustomSymbolStrType')]">
        <xsl:call-template name="copyWithPrefix"><xsl:with-param name="cProfile" select="'ert'" /></xsl:call-template>
    </xsl:template>

    <xsl:template match="*[contains(name(), '.CustomSymbolStrField')]">
        <xsl:call-template name="copyWithPrefix"><xsl:with-param name="cProfile" select="'ert'" /></xsl:call-template>
    </xsl:template>

    <xsl:template match="*[contains(name(), '.CustomSymbolStrFcn')]">
        <xsl:call-template name="copyWithPrefix"><xsl:with-param name="cProfile" select="'ert'" /></xsl:call-template>
    </xsl:template>

    <xsl:template match="*[contains(name(), '.CustomSymbolStrTmpVar')]">
        <xsl:call-template name="copyWithPrefix"><xsl:with-param name="cProfile" select="'ert'" /></xsl:call-template>
    </xsl:template>

    <xsl:template match="*[contains(name(), '.CustomSymbolStrMacro')]">
        <xsl:call-template name="copyWithPrefix"><xsl:with-param name="cProfile" select="'ert'" /></xsl:call-template>
    </xsl:template>

    <xsl:template match="*[contains(name(), '.CustomSymbolStrEMXArray')]">
        <xsl:call-template name="copyWithPrefix"><xsl:with-param name="cProfile" select="'ert'" /></xsl:call-template>
    </xsl:template>

    <xsl:template match="*[contains(name(), '.MaxIdLength')]">
        <xsl:call-template name="copyWithPrefix" />
    </xsl:template>

    <xsl:template match="*[contains(name(), '.ReservedNameArray')]">
        <xsl:call-template name="copyWithPrefix"></xsl:call-template>
    </xsl:template>

    <xsl:template match="*[contains(name(), '.CustomSourceCode')]">
        <xsl:call-template name="copyWithPrefix"><xsl:with-param name="extraPrefix" select="'.customcode'" /></xsl:call-template>
    </xsl:template>

    <xsl:template match="*[contains(name(), '.CustomHeaderCode')]">
        <xsl:call-template name="copyWithPrefix"><xsl:with-param name="extraPrefix" select="'.customcode'" /></xsl:call-template>
    </xsl:template>

    <xsl:template match="*[contains(name(), '.CustomInitializer')]">
        <xsl:call-template name="copyWithPrefix"><xsl:with-param name="extraPrefix" select="'.customcode'" /></xsl:call-template>
    </xsl:template>

    <xsl:template match="*[contains(name(), '.CustomTerminator')]">
        <xsl:call-template name="copyWithPrefix"><xsl:with-param name="extraPrefix" select="'.customcode'" /></xsl:call-template>
    </xsl:template>

    <xsl:template match="*[contains(name(), '.CustomInclude')]">
        <xsl:call-template name="copyWithPrefix"><xsl:with-param name="extraPrefix" select="'.customcode'" /></xsl:call-template>
    </xsl:template>

    <xsl:template match="*[contains(name(), '.CustomSource') and not(contains(name(), '.CustomSourceCode'))]">
        <xsl:call-template name="copyWithPrefix"><xsl:with-param name="extraPrefix" select="'.customcode'" /></xsl:call-template>
    </xsl:template>

    <xsl:template match="*[contains(name(), '.CustomLibrary')]">
        <xsl:call-template name="copyWithPrefix"><xsl:with-param name="extraPrefix" select="'.customcode'" /></xsl:call-template>
    </xsl:template>

    <xsl:template match="*[contains(name(), '.PostCodeGenCommand')]">
        <xsl:call-template name="copyWithPrefix" />
    </xsl:template>

    <xsl:template match="*[contains(name(), '.EnableMemcpy')]">
        <xsl:call-template name="copyWithPrefix" />
    </xsl:template>

    <xsl:template match="*[contains(name(), '.MemcpyThreshold')]">
        <xsl:call-template name="copyWithPrefix" />
    </xsl:template>

    <xsl:template match="*[contains(name(), '.InitFltsAndDblsToZero')]">
        <xsl:call-template name="copyWithPrefix" />
    </xsl:template>

    <xsl:template match="*[contains(name(), '.InlineThreshold') and not(contains(name(), '.InlineThresholdMax'))]">
        <xsl:call-template name="copyWithPrefix" />
    </xsl:template>

    <xsl:template match="*[contains(name(), '.InlineThresholdMax')]">
        <xsl:call-template name="copyWithPrefix" />
    </xsl:template>

    <xsl:template match="*[contains(name(), '.InlineStackLimit')]">
        <xsl:call-template name="copyWithPrefix" />
    </xsl:template>

    <xsl:template match="*[contains(name(), '.StackUsageMax')]">
        <xsl:call-template name="copyWithPrefix" />
    </xsl:template>

    <xsl:template match="*[contains(name(), '.ConstantFoldingTimeout')]">
        <xsl:call-template name="copyWithPrefix" />
    </xsl:template>

    <xsl:template match="*[contains(name(), '.CCompilerOptimization')]">
        <xsl:call-template name="copyWithPrefix" />
    </xsl:template>

    <xsl:template match="*[contains(name(), '.CCompilerCustomOptimizations')]">
        <xsl:call-template name="copyWithPrefix" />
    </xsl:template>

    <xsl:template match="*[contains(name(), '.GenerateMakefile')]">
        <xsl:call-template name="copyWithPrefix"><xsl:with-param name="extraPrefix" select="'.general'" /></xsl:call-template>
    </xsl:template>

    <xsl:template match="*[contains(name(), '.BuildToolEnable')]">
        <xsl:call-template name="copyWithPrefix">
            <xsl:with-param name="extraPrefix" select="'.buildtool'" />
            <xsl:with-param name="useProfile" select="false" />
        </xsl:call-template>
    </xsl:template>

    <xsl:template match="*[contains(name(), '.BuildToolConfiguration')]">
        <xsl:call-template name="copyWithPrefix">
            <xsl:with-param name="extraPrefix" select="'.buildtool'" />
            <xsl:with-param name="useProfile" select="false" />
        </xsl:call-template>
    </xsl:template>

    <xsl:template match="*[contains(name(), '.MakeCommand')]">
        <xsl:call-template name="copyWithPrefix"><xsl:with-param name="extraPrefix" select="'.general'" /></xsl:call-template>
    </xsl:template>

    <xsl:template match="*[contains(name(), '.TemplateMakefile')]">
        <xsl:call-template name="copyWithPrefix"><xsl:with-param name="extraPrefix" select="'.general'" /></xsl:call-template>
    </xsl:template>

    <xsl:template match="*[contains(name(), '.TargetFunctionLibrary')]">
        <xsl:call-template name="copyWithPrefix"><xsl:with-param name="cProfile" select="'grt'" /></xsl:call-template>
        <xsl:call-template name="copyWithPrefix"><xsl:with-param name="cProfile" select="'ert'" /></xsl:call-template>
    </xsl:template>

    <xsl:template match="*[contains(name(), '.CodeReplacementLibrary')]">
        <xsl:call-template name="copyWithPrefix"><xsl:with-param name="cProfile" select="'grt'" /></xsl:call-template>
        <xsl:call-template name="copyWithPrefix"><xsl:with-param name="cProfile" select="'ert'" /></xsl:call-template>
    </xsl:template>
    
    <xsl:template match="*[contains(name(), '.PurelyIntegerCode')]">
        <xsl:call-template name="copyWithPrefix"><xsl:with-param name="cProfile" select="'ert'" /></xsl:call-template>
    </xsl:template>

    <xsl:template match="*[contains(name(), '.SupportNonFinite')]">
        <xsl:call-template name="copyWithPrefix"><xsl:with-param name="cProfile" select="'ert'" /></xsl:call-template>
        <xsl:call-template name="copyWithPrefix"><xsl:with-param name="cProfile" select="'grt'" /></xsl:call-template>
    </xsl:template>

    <xsl:template match="*[contains(name(), '.IncludeTerminateFcn')]">
        <xsl:call-template name="copyWithPrefix"><xsl:with-param name="cProfile" select="'ert'" /></xsl:call-template>
    </xsl:template>

    <xsl:template match="*[contains(name(), '.MultiInstanceCode')]">
        <xsl:call-template name="copyWithPrefix"><xsl:with-param name="cProfile" select="'ert'" /></xsl:call-template>
    </xsl:template>

    <xsl:template match="*[contains(name(), '.ParenthesesLevel')]">
        <xsl:call-template name="copyWithPrefix"><xsl:with-param name="cProfile" select="'ert'" /></xsl:call-template>
    </xsl:template>

    <xsl:template match="*[contains(name(), '.ConvertIfToSwitch')]">
        <xsl:call-template name="copyWithPrefix"><xsl:with-param name="cProfile" select="'ert'" /></xsl:call-template>
    </xsl:template>

    <xsl:template match="*[contains(name(), '.PreserveExternInFcnDecls')]">
        <xsl:call-template name="copyWithPrefix"><xsl:with-param name="cProfile" select="'ert'" /></xsl:call-template>
    </xsl:template>

    <xsl:template match="*[contains(name(), '.WorkingFolder')]">
        <xsl:variable name="oldName">
            <xsl:choose>
                <xsl:when test="count(ancestor::*[@key = 'profile.mex']) = 1">param.mex.paths.working</xsl:when>
                <xsl:otherwise>param.grt.paths.working</xsl:otherwise>
            </xsl:choose>
        </xsl:variable>

        <xsl:element name="{$oldName}">
            <xsl:choose>
                <xsl:when test=". = 'option.WorkingFolder.Project'">option.paths.working.project</xsl:when>
                <xsl:when test=". = 'option.WorkingFolder.Current'">option.paths.working.current</xsl:when>
                <xsl:when test=". = 'option.WorkingFolder.Specified'">option.paths.working.specified</xsl:when>
            </xsl:choose>
        </xsl:element>
    </xsl:template>

    <xsl:template match="*[contains(name(), '.SpecifiedWorkingFolder')]">
        <xsl:variable name="oldName">
            <xsl:choose>
                <xsl:when test="count(ancestor::*[@key = 'profile.mex']) = 1">param.mex.paths.working.specified</xsl:when>
                <xsl:otherwise>param.grt.paths.working.specified</xsl:otherwise>
            </xsl:choose>
        </xsl:variable>

        <xsl:element name="{$oldName}">
            <xsl:copy-of select="node()" />
        </xsl:element>
    </xsl:template>

    <xsl:template match="*[contains(name(), '.BuildFolder')]">
        <xsl:variable name="oldName">
            <xsl:choose>
                <xsl:when test="count(ancestor::*[@key = 'profile.mex']) = 1">param.mex.paths.build</xsl:when>
                <xsl:otherwise>param.grt.paths.build</xsl:otherwise>
            </xsl:choose>
        </xsl:variable>

        <xsl:element name="{$oldName}">
            <xsl:choose>
                <xsl:when test=". = 'option.BuildFolder.Project'">option.paths.build.project</xsl:when>
                <xsl:when test=". = 'option.BuildFolder.Current'">option.paths.build.current</xsl:when>
                <xsl:when test=". = 'option.BuildFolder.Specified'">option.paths.build.specified</xsl:when>
            </xsl:choose>
        </xsl:element>
    </xsl:template>

    <xsl:template match="*[contains(name(), '.SpecifiedBuildFolder')]">
        <xsl:variable name="oldName">
            <xsl:choose>
                <xsl:when test="count(ancestor::*[@key = 'profile.mex']) = 1">param.mex.paths.build.specified</xsl:when>
                <xsl:otherwise>param.grt.paths.build.specified</xsl:otherwise>
            </xsl:choose>
        </xsl:variable>

        <xsl:element name="{$oldName}">
            <xsl:copy-of select="node()" />
        </xsl:element>
    </xsl:template>

    <xsl:template match="*[contains(name(), '.SearchPaths')]">
        <xsl:variable name="oldName">
            <xsl:choose>
                <xsl:when test="count(ancestor::*[@key = 'profile.mex']) = 1">param.mex.paths.search</xsl:when>
                <xsl:otherwise>param.grt.paths.search</xsl:otherwise>
            </xsl:choose>
        </xsl:variable>

        <xsl:element name="{$oldName}">
            <xsl:copy-of select="node()" />
        </xsl:element>
    </xsl:template>

    <!-- identity transformation for everything else -->
    <xsl:template match="@*|node()">
        <xsl:copy>
            <xsl:apply-templates select="@*|node()"/>
        </xsl:copy>
    </xsl:template>

    <!-- ============== Helper templates ============== -->

    <!-- Copy a param and add a prefix in addition to the profile -->

    <xsl:template name="copyWithPrefix">
        <xsl:param name="extraPrefix" select="''" />
        <xsl:param name="cProfile" select="'grt'" />
        <xsl:param name="useProfile" select="true" />

        <xsl:variable name="baseName">
            <xsl:value-of select="substring-after(name(), 'param')" />
        </xsl:variable>

        <xsl:variable name="oldName">
            <xsl:choose>
                <xsl:when test="$useProfile = false">param</xsl:when>
                <xsl:when test="count(ancestor::*[@key = 'profile.mex']) = 1">param.mex</xsl:when>
                <xsl:otherwise>param.<xsl:value-of select="$cProfile" /></xsl:otherwise>
            </xsl:choose>
            <xsl:value-of select="$extraPrefix" />

            <!-- CodeReplacementLibrary was called TargetFunctionLibrary prior to R2012a -->
            <xsl:choose>
                <xsl:when test="$baseName = '.CodeReplacementLibrary'">.TargetFunctionLibrary</xsl:when>
                <xsl:otherwise><xsl:value-of select="$baseName" /></xsl:otherwise>
            </xsl:choose>
        </xsl:variable>

        <xsl:variable name="unset" select="count(ancestor::*[name() = 'unset']) != 0" />

        <xsl:element name="{$oldName}">
            <xsl:choose>
                <!-- Some enum values go from option.foo to option.general.foo -->
                <xsl:when test="not($unset) and (name() = 'param.TargetLang' or name() = 'param.FilePartitionMethod' or name() = 'param.DynamicMemoryAllocation' or name() = 'param.GlobalDataSyncMethod')">
                    <xsl:value-of select="concat('option.general.', substring-after(text(), 'option.'))" />
                </xsl:when>

                <xsl:otherwise>
                    <xsl:copy-of select="node()" />
                </xsl:otherwise>
            </xsl:choose>
        </xsl:element>
    </xsl:template>

    <!-- Translate one generic hardware-related field -->
    <xsl:template name="hardwareField">
        <xsl:param name="oldName" />
        <xsl:param name="newName" />
        <xsl:element name="{$oldName}"><xsl:value-of select="profile[@key = 'profile.c']/*[name() = $newName]" /></xsl:element>
    </xsl:template>

    <!-- Translate one hardware size-related field -->
    <xsl:template name="hardwareSizeField">
        <xsl:param name="oldPrefix" />
        <xsl:param name="newSuffix" />
        <xsl:param name="typeName" />

        <xsl:call-template name="hardwareField">
            <xsl:with-param name="oldName" select="concat($oldPrefix, 'BitPer', $typeName)" />
            <xsl:with-param name="newName" select="concat('param.HardwareSize', $typeName, '.', $newSuffix)" />
        </xsl:call-template>
    </xsl:template>

    <!-- Translate all the hardware fields for target or production -->
    <xsl:template name="hardware">
        <xsl:param name="oldPrefix" />
        <xsl:param name="newSuffix" />

		<xsl:choose>
		    <!-- NEC has been renamed to Renesas -->
		    <xsl:when test="$oldPrefix = 'Target' and //param.HardwareVendor.Target = 'Renesas' and //param.HardwareType.Target = 'V850'">
		    	<TargetVendorName>NEC</TargetVendorName>
		    	<TargetTypeName>V850</TargetTypeName>
		    </xsl:when>
		    
		    <xsl:when test="$oldPrefix = 'Prod' and //param.HardwareVendor.Production = 'Renesas' and //param.HardwareType.Production = 'V850'">
		    	<ProdVendorName>NEC</ProdVendorName>
		    	<ProdTypeName>V850</ProdTypeName>
		    </xsl:when>
		    
		    <xsl:otherwise>
		        <xsl:call-template name="hardwareField">
        		    <xsl:with-param name="oldName" select="concat($oldPrefix, 'VendorName')" />
		            <xsl:with-param name="newName" select="concat('param.HardwareVendor.', $newSuffix)" />
		        </xsl:call-template>
		    </xsl:otherwise>
		</xsl:choose>

        <xsl:call-template name="hardwareField">
            <xsl:with-param name="oldName" select="concat($oldPrefix, 'TypeName')" />
            <xsl:with-param name="newName" select="concat('param.HardwareType.', $newSuffix)" />
        </xsl:call-template>

        <xsl:call-template name="hardwareSizeField">
            <xsl:with-param name="typeName" select="'Char'" />
            <xsl:with-param name="oldPrefix" select="$oldPrefix" />
            <xsl:with-param name="newSuffix" select="$newSuffix" />
        </xsl:call-template>

        <xsl:call-template name="hardwareSizeField">
            <xsl:with-param name="typeName" select="'Short'" />
            <xsl:with-param name="oldPrefix" select="$oldPrefix" />
            <xsl:with-param name="newSuffix" select="$newSuffix" />
        </xsl:call-template>

        <xsl:call-template name="hardwareSizeField">
            <xsl:with-param name="typeName" select="'Int'" />
            <xsl:with-param name="oldPrefix" select="$oldPrefix" />
            <xsl:with-param name="newSuffix" select="$newSuffix" />
        </xsl:call-template>

        <xsl:call-template name="hardwareSizeField">
            <xsl:with-param name="typeName" select="'Long'" />
            <xsl:with-param name="oldPrefix" select="$oldPrefix" />
            <xsl:with-param name="newSuffix" select="$newSuffix" />
        </xsl:call-template>

        <xsl:call-template name="hardwareSizeField">
            <xsl:with-param name="typeName" select="'Float'" />
            <xsl:with-param name="oldPrefix" select="$oldPrefix" />
            <xsl:with-param name="newSuffix" select="$newSuffix" />
        </xsl:call-template>

        <xsl:call-template name="hardwareSizeField">
            <xsl:with-param name="typeName" select="'Double'" />
            <xsl:with-param name="oldPrefix" select="$oldPrefix" />
            <xsl:with-param name="newSuffix" select="$newSuffix" />
        </xsl:call-template>

        <xsl:call-template name="hardwareSizeField">
            <xsl:with-param name="typeName" select="'Pointer'" />
            <xsl:with-param name="oldPrefix" select="$oldPrefix" />
            <xsl:with-param name="newSuffix" select="$newSuffix" />
        </xsl:call-template>

        <xsl:call-template name="hardwareField">
            <xsl:with-param name="oldName" select="concat($oldPrefix, 'WordSize')" />
            <xsl:with-param name="newName" select="concat('param.HardwareSizeWord.', $newSuffix)" />
        </xsl:call-template>

        <xsl:element name="{concat($oldPrefix, 'Endianess')}">
            <xsl:variable name="endianness" select="profile[@key = 'profile.c']/*[name() = concat('param.HardwareEndianness.', $newSuffix)]" />

            <xsl:choose>
                <xsl:when test="$endianness = 'option.HardwareEndianness.Unspecified'">0</xsl:when>
                <xsl:when test="$endianness = 'option.HardwareEndianness.Little'">1</xsl:when>
                <xsl:when test="$endianness = 'option.HardwareEndianness.Big'">2</xsl:when>
            </xsl:choose>
        </xsl:element>

        <xsl:element name="{concat($oldPrefix, 'LargestAtomicInteger')}">
            <xsl:variable name="atomicIntegerSize" select="profile[@key = 'profile.c']/*[name() = concat('param.HardwareAtomicIntegerSize.', $newSuffix)]" />

            <xsl:choose>
                <xsl:when test="$atomicIntegerSize = 'option.HardwareAtomicIntegerSize.Char'">0</xsl:when>
                <xsl:when test="$atomicIntegerSize = 'option.HardwareAtomicIntegerSize.Short'">1</xsl:when>
                <xsl:when test="$atomicIntegerSize = 'option.HardwareAtomicIntegerSize.Int'">2</xsl:when>
                <xsl:when test="$atomicIntegerSize = 'option.HardwareAtomicIntegerSize.Long'">3</xsl:when>
            </xsl:choose>
        </xsl:element>

        <xsl:element name="{concat($oldPrefix, 'LargestAtomicFloat')}">
            <xsl:variable name="atomicFloatSize" select="profile[@key = 'profile.c']/*[name() = concat('param.HardwareAtomicFloatSize.', $newSuffix)]" />

            <xsl:choose>
                <xsl:when test="$atomicFloatSize = 'option.HardwareAtomicFloatSize.None'">0</xsl:when>
                <xsl:when test="$atomicFloatSize = 'option.HardwareAtomicFloatSize.Float'">1</xsl:when>
                <xsl:when test="$atomicFloatSize = 'option.HardwareAtomicFloatSize.Double'">2</xsl:when>
            </xsl:choose>
        </xsl:element>

        <xsl:element name="{concat($oldPrefix, 'IntDivRoundTo')}">
            <xsl:variable name="divisionRounding" select="profile[@key = 'profile.c']/*[name() = concat('param.HardwareDivisionRounding.', $newSuffix)]" />

            <xsl:choose>
                <xsl:when test="$divisionRounding = 'option.HardwareDivisionRounding.Undefined'">0</xsl:when>
                <xsl:when test="$divisionRounding = 'option.HardwareDivisionRounding.Zero'">1</xsl:when>
                <xsl:when test="$divisionRounding = 'option.HardwareDivisionRounding.Floor'">2</xsl:when>
            </xsl:choose>
        </xsl:element>

        <xsl:call-template name="hardwareField">
            <xsl:with-param name="oldName" select="concat($oldPrefix, 'ShiftRightIntArith')" />
            <xsl:with-param name="newName" select="concat('param.HardwareArithmeticRightShift.', $newSuffix)" />
        </xsl:call-template>
    </xsl:template>

</xsl:stylesheet>
