/* -------------------------------------------------------------------- */
/* ---------  N R L M S I S E - 0 0    M O D E L    2 0 0 1  ---------- */
/* -------------------------------------------------------------------- */

/* This file is part of the NRLMSISE-00  C source code package - release
 * 20041227
 *
 * The NRLMSISE-00 model was developed by Mike Picone, Alan Hedin, and
 * Doug Drob. They also wrote a NRLMSISE-00 distribution package in
 * FORTRAN which is available at
 * http://uap-www.nrl.navy.mil/models_web/msis/msis_home.htm
 *
 * This work is derived from the version Dominik Brodowski implemented in C. 
 * Modifications were made to the code to more easily work with mex and 
 * S-functions. See http://www.brodo.de/english/pub/nrlmsise/index.html for
 * details on the original C version.
 */

/* Copyright 2007-2013 The MathWorks, Inc. */

#define DGTR 0.0174533
#define RGAS 831.4
#define DR   0.0172142
#define SR   0.000072722
#define HR   0.2618

/* ------------------------------------------------------------------- */
/* ------------------------------ INCLUDES --------------------------- */
/* ------------------------------------------------------------------- */

#include "export/include/aeronrlmsise/aeroatmnrlmsise00.hpp"   /* header for nrlmsise-00 */
#include <math.h>     /* maths functions */

/* ------------------------------------------------------------------- */
/* ------------------------- SHARED VARIABLES ------------------------ */
/* ------------------------------------------------------------------- */


/* POWER7 */
extern double pt[150];
extern double pd[9][150];
extern double ps[150];
extern double pdl[2][25];
extern double ptl[4][100];
extern double pma[10][100];
extern double sam[100];

/* LOWER7 */
extern double ptm[10];
extern double pdm[8][10];
extern double pavgm[10];

/* LPOLY */
static double dfa;
static double plg[4][9];



/* ------------------------------------------------------------------- */
/* ------------------------------ TSELEC ----------------------------- */
/* ------------------------------------------------------------------- */

void tselec(struct nrlmsise_flags *flags) {
    int i;
    for (i=0;i<24;i++) {
        if (i!=9) {
            if (flags->switches[i]==1)
                flags->sw[i]=1;
            else
                flags->sw[i]=0;
            if (flags->switches[i]>0)
                flags->swc[i]=1;
            else
                flags->swc[i]=0;
        } else {
            flags->sw[i]=flags->switches[i];
            flags->swc[i]=flags->switches[i];
        }
    }
}



/* ------------------------------------------------------------------- */
/* ------------------------------ GLATF ------------------------------ */
/* ------------------------------------------------------------------- */

void glatf(const double lat, double *gv, double *reff, double flag) {
    double c2;
    double xlat;

/* Latitude variation of gravity (none for flag = 0) */
    xlat = lat;
    if ( flag == 0 ) {
        xlat = 45.0;
    }
    c2 = cos(2.0 * DGTR * xlat);
    *gv = 980.616 * (1.0 - 0.0026373 * c2);
    *reff = 2.0 * (*gv) / (3.085462E-6 + 2.27E-9 * c2) * 1.0E-5;
}



/* ------------------------------------------------------------------- */
/* ------------------------------ CCOR ------------------------------- */
/* ------------------------------------------------------------------- */

double ccor(double alt, double r, double h1, double zh) {
/*        CHEMISTRY/DISSOCIATION CORRECTION FOR MSIS MODELS
 *         ALT - altitude
 *         R - target ratio
 *         H1 - transition scale length
 *         ZH - altitude of 1/2 R
 */
    double e, ex;

    e = (alt - zh) / h1;

    if (e > 70){
        e = 0;
    }
    else{
        if (e < -70){
            e = r;
        }
        else {
            ex = exp(e);
            e = r / (1.0 + ex);
        }
    }
    return exp(e);
}



/* ------------------------------------------------------------------- */
/* ------------------------------ CCOR ------------------------------- */
/* ------------------------------------------------------------------- */

double ccor2(double alt, double r, double h1, double zh, double h2) {
/*       O&O2 CHEMISTRY/DISSOCIATION CORRECTION FOR MSIS MODELS
 *         ALT - altitude
 *         R - target ratio
 *         H1 - transition scale length
 *         ZH - altitude of 1/2 R
 *         H2 - transition scale length #2 ?
 */
    double e1, e2, ex1, ex2, ccor2v;

    e1 = (alt - zh) / h1;
    e2 = (alt - zh) / h2;

    if ((e1 > 70) || (e2 > 70)){
        ccor2v = 0;
    }
    else{
        if ((e1 < -70) && (e2 < -70)){
            ccor2v = r ;
	}
        else {
            ex1 = exp(e1);
            ex2 = exp(e2);
            ccor2v = r / (1.0 + 0.5 * (ex1 + ex2));
        }
    }
    return exp(ccor2v);
}



/* ------------------------------------------------------------------- */
/* ------------------------------- SCALH ----------------------------- */
/* ------------------------------------------------------------------- */

double scalh(double alt, double xm, double temp, double gsurf, double re) {
    double g, denom;

    denom = 1.0 + alt / re; 
    g = gsurf / ( denom * denom ); 
    g = RGAS * temp / (g * xm);
    return g;
}



/* ------------------------------------------------------------------- */
/* -------------------------------- DNET ----------------------------- */
/* ------------------------------------------------------------------- */

double dnet (double dd, double dm, double zhm, double xmm, double xm) {
/*       TURBOPAUSE CORRECTION FOR MSIS MODELS
 *        Root mean density
 *         DD - diffusive density
 *         DM - full mixed density
 *         ZHM - transition scale length
 *         XMM - full mixed molecular weight
 *         XM  - species molecular weight
 *         DNET - combined density
 */
    double a;
    double ylog;
    
    a  = zhm / (xmm - xm);
    if (!((dm > 0) && (dd > 0))) {
        if ((dd == 0) && (dm == 0)){
            dd = 1;
        }
        if (dm == 0){
            return dd;
        }
        if (dd == 0){
            return dm;
        }
    } 
    ylog = a * log( dm / dd );
    if (ylog < -10){
        return dd;
    }
    if (ylog > 10){
        return dm;
    }
    a = dd * pow((1.0 + exp(ylog)),(1.0 / a));
    return a;
}



/* ------------------------------------------------------------------- */
/* ------------------------------- SPLINI ---------------------------- */
/* ------------------------------------------------------------------- */

void splini (double *xa, double *ya, double *y2a, int n, double x, double *y) {
/*      INTEGRATE CUBIC SPLINE FUNCTION FROM XA(1) TO X
 *       XA,YA: ARRAYS OF TABULATED FUNCTION IN ASCENDING ORDER BY X
 *       Y2A: ARRAY OF SECOND DERIVATIVES
 *       N: SIZE OF ARRAYS XA,YA,Y2A
 *       X: ABSCISSA ENDPOINT FOR INTEGRATION
 *       Y: OUTPUT VALUE
 */
    double yi=0;
    int klo=0;
    int khi=1;
    double xx, h, a, b, a2, b2;
    while ((x>xa[klo]) && (khi<n)) {
        xx=x;
        if (khi<(n-1)) {
            if (x<xa[khi])
                xx=x;
            else 
                xx=xa[khi];
        }
        h = xa[khi] - xa[klo];
        a = (xa[khi] - xx)/h;
        b = (xx - xa[klo])/h;
        a2 = a*a;
        b2 = b*b;
        yi += ((1.0 - a2) * ya[klo] / 2.0 + b2 * ya[khi] / 2.0 + ((-(1.0+a2*a2)/4.0 + a2/2.0) * y2a[klo] + (b2*b2/4.0 - b2/2.0) * y2a[khi]) * h * h / 6.0) * h;
        klo++;
        khi++;
    }
    *y = yi;
}



/* ------------------------------------------------------------------- */
/* ------------------------------- SPLINT ---------------------------- */
/* ------------------------------------------------------------------- */

void splint (double *xa, double *ya, double *y2a, int n, double x, double *y) {
/*      CALCULATE CUBIC SPLINE INTERP VALUE
 *       ADAPTED FROM NUMERICAL RECIPES BY PRESS ET AL.
 *       XA,YA: ARRAYS OF TABULATED FUNCTION IN ASCENDING ORDER BY X
 *       Y2A: ARRAY OF SECOND DERIVATIVES
 *       N: SIZE OF ARRAYS XA,YA,Y2A
 *       X: ABSCISSA FOR INTERPOLATION
 *       Y: OUTPUT VALUE
 */
    int klo=0;
    int khi=n-1;
    int k;
    double h;
    double a, b, yi;
    
    while ((khi-klo)>1) {
        k=(khi+klo)/2;
        if (xa[k]>x)
            khi=k;
        else
            klo=k;
    }
    h = xa[khi] - xa[klo];
    a = (xa[khi] - x)/h;
    b = (x - xa[klo])/h;
    yi = a * ya[klo] + b * ya[khi] + ((a*a*a - a) * y2a[klo] + (b*b*b - b) * y2a[khi]) * h * h/6.0;
    *y = yi;
}



/* ------------------------------------------------------------------- */
/* ------------------------------- SPLINE ---------------------------- */
/* ------------------------------------------------------------------- */

void spline (double *x, double *y, int n, double yp1, double ypn, double *y2) {
/*       CALCULATE 2ND DERIVATIVES OF CUBIC SPLINE INTERP FUNCTION
 *       ADAPTED FROM NUMERICAL RECIPES BY PRESS ET AL
 *       X,Y: ARRAYS OF TABULATED FUNCTION IN ASCENDING ORDER BY X
 *       N: SIZE OF ARRAYS X,Y
 *       YP1,YPN: SPECIFIED DERIVATIVES AT X[0] AND X[N-1]; VALUES
 *                >= 1E30 SIGNAL SECOND DERIVATIVE ZERO
 *       Y2: OUTPUT ARRAY OF SECOND DERIVATIVES
 */
    double u[10];
    double sig, p, qn, un;
    int i, k;

    if (yp1>0.99E30) {
        y2[0]=0;
        u[0]=0;
    } else {
        y2[0]=-0.5;
        u[0]=(3.0/(x[1]-x[0]))*((y[1]-y[0])/(x[1]-x[0])-yp1);
    }
    for (i=1;i<(n-1);i++) {
        sig = (x[i]-x[i-1])/(x[i+1] - x[i-1]);
        p = sig * y2[i-1] + 2.0;
        y2[i] = (sig - 1.0) / p;
        u[i] = (6.0 * ((y[i+1] - y[i])/(x[i+1] - x[i]) -(y[i] - y[i-1]) / (x[i] - x[i-1]))/(x[i+1] - x[i-1]) - sig * u[i-1])/p;
    }
    if (ypn>0.99E30) {
        qn = 0;
        un = 0;
    } else {
        qn = 0.5;
        un = (3.0 / (x[n-1] - x[n-2])) * (ypn - (y[n-1] - y[n-2])/(x[n-1] - x[n-2]));
    }
    y2[n-1] = (un - qn * u[n-2]) / (qn * y2[n-2] + 1.0);
    for (k=n-2;k>=0;k--)
        y2[k] = y2[k] * y2[k+1] + u[k];

}



/* ------------------------------------------------------------------- */
/* ------------------------------- DENSM ----------------------------- */
/* ------------------------------------------------------------------- */

__inline_double zeta(double zz, double zl, double re) {
    return ((zz - zl)*(re + zl)/(re + zz)); 
}

double densm (double alt, double d0, double xm, double *tz, int mn3, 
			  double *zn3, double *tn3, double *tgn3, int mn2, double *zn2, 
			  double *tn2, double *tgn2, double gsurf, double re) {
/*      Calculate Temperature and Density Profiles for lower atmos.  */
    double xs[10], ys[10], y2out[10];
    double z, z1, z2, t1, t2, zg, zgdif;
    double yd1, yd2;
    double x, y, yi;
    double expll, gamm, glb, sqterm;
    double densm_tmp;
    int mn;
    int k;
    densm_tmp=d0;
    if (alt>zn2[0]) {
        if (xm==0.0)
            return *tz;
        else
            return d0;
    }

    /* STRATOSPHERE/MESOSPHERE TEMPERATURE */
    if (alt>zn2[mn2-1])
        z=alt;
    else
        z=zn2[mn2-1];
    mn=mn2;
    z1=zn2[0];
    z2=zn2[mn-1];
    t1=tn2[0];
    t2=tn2[mn-1];
    zg = zeta(z, z1, re);
    zgdif = zeta(z2, z1, re);

    /* set up spline nodes */
    for (k=0;k<mn;k++) {
        xs[k]=zeta(zn2[k],z1,re)/zgdif;
        ys[k]=1.0 / tn2[k];
    }
    yd1=-tgn2[0] / (t1*t1) * zgdif;
    sqterm = (re+z2)/(re+z1);
    yd2=-tgn2[1] / (t2*t2) * zgdif * (sqterm * sqterm); 

    /* calculate spline coefficients */
    spline (xs, ys, mn, yd1, yd2, y2out);
    x = zg/zgdif;
    splint (xs, ys, y2out, mn, x, &y);

    /* temperature at altitude */
    *tz = 1.0 / y;
    if (xm!=0.0) {
        /* calculate stratosphere / mesosphere density */
        sqterm = 1.0 + z1/re;
        glb = gsurf / (sqterm * sqterm);
        gamm = xm * glb * zgdif / RGAS;

        /* Integrate temperature profile */
        splini(xs, ys, y2out, mn, x, &yi);
        expll=gamm*yi;
        if (expll>50.0)
            expll=50.0;

        /* Density at altitude */
        densm_tmp = densm_tmp * (t1 / *tz) * exp(-expll);
    }

    if (alt>zn3[0]) {
        if (xm==0.0)
            return *tz;
        else
            return densm_tmp;
    }

    /* troposphere / stratosphere temperature */
    z = alt;
    mn = mn3;
    z1=zn3[0];
    z2=zn3[mn-1];
    t1=tn3[0];
    t2=tn3[mn-1];
    zg=zeta(z, z1, re);
    zgdif=zeta(z2, z1, re);

    /* set up spline nodes */
    for (k=0;k<mn;k++) {
        xs[k] = zeta(zn3[k], z1, re) / zgdif;
        ys[k] = 1.0 / tn3[k];
    }
    yd1=-tgn3[0] / (t1*t1) * zgdif;
    sqterm = (re+z2)/(re+z1);
    yd2=-tgn3[1] / (t2*t2) * zgdif * (sqterm * sqterm); 

    /* calculate spline coefficients */
    spline (xs, ys, mn, yd1, yd2, y2out);
    x = zg/zgdif;
    splint (xs, ys, y2out, mn, x, &y);

    /* temperature at altitude */
    *tz = 1.0 / y;
    if (xm!=0.0) {
        /* calculate tropospheric / stratosphere density */
        sqterm = 1.0 + z1/re;
        glb = gsurf / (sqterm * sqterm); 
        gamm = xm * glb * zgdif / RGAS;

        /* Integrate temperature profile */
        splini(xs, ys, y2out, mn, x, &yi);
        expll=gamm*yi;
        if (expll>50.0)
            expll=50.0;

        /* Density at altitude */
        densm_tmp = densm_tmp * (t1 / *tz) * exp(-expll);
    }
    if (xm==0.0)
        return *tz;
    else
        return densm_tmp;
}



/* ------------------------------------------------------------------- */
/* ------------------------------- DENSU ----------------------------- */
/* ------------------------------------------------------------------- */

double densu (double alt, double dlb, double tinf, double tlb, double xm, 
			  double alpha, double *tz, double zlb, double s2, int mn1, 
			  double *zn1, double *tn1, double *tgn1, double gsurf, 
			  double re) {
/*      Calculate Temperature and Density Profiles for MSIS models
 *      New lower thermo polynomial
 */
    double yd2, yd1, y;
    double densu_temp = 1.0;
    double za, z, zg2, tt, ta;
    double dta, z2, t2, zg;
    double x = 0.0;
    double z1 = 0.0;
    double t1 = 0.0;
    double zgdif = 0.0;
    int mn = 0;
    int k;
    double glb;
    double expll;
    double yi;
    double densa, sqterm;
    double gammad, gamm;
    double xs[5], ys[5], y2out[5];
    /* joining altitudes of Bates and spline */
    za=zn1[0];
    if (alt>za)
        z=alt;
    else
        z=za;

    /* geopotential altitude difference from ZLB */
    zg2 = zeta(z, zlb, re);

    /* Bates temperature */
    tt = tinf - (tinf - tlb) * exp(-s2*zg2);
    ta = tt;
    *tz = tt;
    densu_temp = *tz;

    if (alt<za) {
        /* calculate temperature below ZA
         * temperature gradient at ZA from Bates profile */
        sqterm = (re+zlb)/(re+za);
        dta = (tinf - ta) * s2 * (sqterm * sqterm); 
        tgn1[0]=dta;
        tn1[0]=ta;
        if (alt>zn1[mn1-1])
            z=alt;
        else
            z=zn1[mn1-1];
        mn=mn1;
        z1=zn1[0];
        z2=zn1[mn-1];
        t1=tn1[0];
        t2=tn1[mn-1];
        /* geopotential difference from z1 */
        zg = zeta (z, z1, re);
        zgdif = zeta(z2, z1, re);
        /* set up spline nodes */
        for (k=0;k<mn;k++) {
            xs[k] = zeta(zn1[k], z1, re) / zgdif;
            ys[k] = 1.0 / tn1[k];
        }
        /* end node derivatives */
        yd1 = -tgn1[0] / (t1*t1) * zgdif;
        sqterm = (re+z2)/(re+z1);
        yd2 = -tgn1[1] / (t2*t2) * zgdif * (sqterm * sqterm); 
        /* calculate spline coefficients */
        spline (xs, ys, mn, yd1, yd2, y2out);
        x = zg / zgdif;
        splint (xs, ys, y2out, mn, x, &y);
        /* temperature at altitude */
        *tz = 1.0 / y;
        densu_temp = *tz;
    }
    if (xm==0)
        return densu_temp;

    /* calculate density above za */
    sqterm = 1.0 + zlb/re;
    glb = gsurf / (sqterm * sqterm); 
    gammad = xm * glb / (s2 * RGAS * tinf);
    expll = exp(-s2 * gammad * zg2);
    if (expll > 50.0){
        expll = 50.0;
    }
    if (tt <= 0){
        expll = 50.0;
    }

    /* density at altitude */
    densa = dlb * pow((tlb/tt),((1.0+alpha+gammad))) * expll;
    densu_temp=densa;
    if (alt>=za)
        return densu_temp;

    /* calculate density below za */
    sqterm = 1.0 + z1/re;
    glb = gsurf / (sqterm * sqterm); 
    gamm = xm * glb * zgdif / RGAS;

    /* integrate spline temperatures */
    splini (xs, ys, y2out, mn, x, &yi);
    expll = gamm * yi;
    if (expll > 50.0){
        expll = 50.0;
    }
    if (*tz <= 0){
        expll = 50.0;
    }

    /* density at altitude */
    densu_temp = densu_temp * pow ((t1 / *tz),(1.0 + alpha)) * exp(-expll);
    return densu_temp;
}



/* ------------------------------------------------------------------- */
/* ------------------------------- GLOBE7 ---------------------------- */
/* ------------------------------------------------------------------- */

/*    3hr Magnetic activity functions */
/*    Eq. A24d */
__inline_double g0(double a, double *p) {
    return (a - 4.0 + (p[25] - 1.0) * (a - 4.0 + (exp(-sqrt(p[24]*p[24]) * (a - 4.0)) - 1.0) / sqrt(p[24]*p[24])));
}

/*    Eq. A24c */
__inline_double sumex(double ex) {
    return (1.0 + (1.0 - pow(ex,19.0)) / (1.0 - ex) * pow(ex,0.5));
}

/*    Eq. A24a */
__inline_double sg0(double ex, double *p, const double *ap) {
    return (g0(ap[1],p) + (g0(ap[2],p)*ex + g0(ap[3],p)*ex*ex + 
                           g0(ap[4],p)*ex*ex*ex + (g0(ap[5],p)*ex*ex*ex*ex + 
                                                   g0(ap[6],p)*pow(ex,12.0))*(1.0-pow(ex,8.0))/(1.0-ex)))/sumex(ex);
}

double globe7(double *p, const int doy, const double sec, double lat, double lon,
			  const double tloc, const double f107a, const double f107, double ap_in, 
			  const double *ap_a, struct nrlmsise_flags *flags, 
			  double *ctloc, double *stloc, double *c2tloc, double *s2tloc, 
			  double *c3tloc, double *s3tloc, double *apdf, double *apt) {
/*       CALCULATE G(L) FUNCTION 
 *       Upper Thermosphere Parameters */

    double t[15];
    int i,j;
    int sw9=1;
    double apd;
    double c, s, c2, c4, s2;
    double cd32, cd18, cd14, cd39;
    double p32, p18, p14, p39;
    double df;
    double f1, f2;
    double tinf;
    const double *ap;

    for (j = 0; j < 14; j++){
        t[j] = 0;
    }
    if (flags->sw[9] > 0){
        sw9=1;
    }
    else if (flags->sw[9]<0){
        sw9=-1;
    }

    /* calculate legendre polynomials */
    c = sin(lat * DGTR);
    s = cos(lat * DGTR);
    c2 = c * c;
    c4 = c2 * c2;
    s2 = s * s;

    plg[0][1] = c;
    plg[0][2] = 0.5 * (3.0 * c2 - 1.0);
    plg[0][3] = 0.5 * (5.0 * c * c2 - 3.0 * c);
    plg[0][4] = (35.0 * c4 - 30.0*c2 + 3.0)/8.0;
    plg[0][5] = (63.0 * c2* c2 * c - 70.0 * c2 * c + 15.0 * c) / 8.0;
    plg[0][6] = (11.0 * c * plg[0][5] - 5.0*plg[0][4])/6.0;
/*      plg[0][7] = (13.0 * c * plg[0][6] - 6.0 * plg[0][5]) / 7.0; */
    plg[1][1] = s;
    plg[1][2] = 3.0 * c * s;
    plg[1][3] = 1.5 * (5.0 * c2 - 1.0) * s;
    plg[1][4] = 2.5 * (7.0 * c2 * c - 3.0 * c) * s;
    plg[1][5] = 1.875 * (21.0 * c4 - 14.0 * c2 + 1.0) * s;
    plg[1][6] = (11.0 * c * plg[1][5] - 6.0 * plg[1][4]) / 5.0;
/*      plg[1][7] = (13.0 * c * plg[1][6] - 7.0 * plg[1][5]) / 6.0; */
/*      plg[1][8] = (15.0 * c * plg[1][7] - 8.0 * plg[1][6]) / 7.0; */
    plg[2][2] = 3.0 * s2;
    plg[2][3] = 15.0 * s2 * c;
    plg[2][4] = 7.5 * (7.0 * c2 -1.0) * s2;
    plg[2][5] = 3.0 * c * plg[2][4] - 2.0 * plg[2][3];
    plg[2][6] =(11.0 * c * plg[2][5] - 7.0 * plg[2][4]) / 4.0;
    plg[2][7] =(13.0 * c * plg[2][6] - 8.0 * plg[2][5]) / 5.0;
    plg[3][3] = 15.0 * s2 * s;
    plg[3][4] = 105.0 * s2 * s * c; 
    plg[3][5] =(9.0 * c * plg[3][4] - 7. * plg[3][3]) / 2.0;
    plg[3][6] =(11.0 * c * plg[3][5] - 8. * plg[3][4]) / 3.0;

    if (!(((flags->sw[7] == 0) && (flags->sw[8] == 0)) && (flags->sw[14] == 0))) {
        *stloc = sin(HR * tloc);
        *ctloc = cos(HR * tloc);
        *s2tloc = sin(2.0 * HR * tloc);
        *c2tloc = cos(2.0 * HR * tloc);
        *s3tloc = sin(3.0 * HR * tloc);
        *c3tloc = cos(3.0 * HR * tloc);
    }

    cd32 = cos(DR * (doy - p[31]));
    cd18 = cos(2.0 * DR * (doy - p[17]));
    cd14 = cos(DR * (doy - p[13]));
    cd39 = cos(2.0 * DR * (doy - p[38]));
    p32  = p[31];
    p18  = p[17];
    p14  = p[13];
    p39  = p[38];

    /* F10.7 EFFECT */
    df   = f107 - f107a;
    dfa  = f107a - 150.0;
    t[0] =  p[19] * df * (1.0 + p[59] * (dfa)) + p[20] * df * df + p[21] * (dfa) + p[29] * (dfa) * (dfa);
    f1   = 1.0 + (p[47] * (dfa) + p[19] * df + p[20] * df * df) * flags->swc[1];
    f2   = 1.0 + (p[49] * (dfa) + p[19] * df + p[20] * df * df) * flags->swc[1];

    /*  TIME INDEPENDENT */
    t[1] = (p[1] * plg[0][2] + p[2] * plg[0][4] + p[22] * plg[0][6]) + 
        (p[14] * plg[0][2]) * (dfa) * flags->swc[1] + p[26] * plg[0][1];

    /*  SYMMETRICAL ANNUAL */
    t[2] = p[18] * cd32;

    /*  SYMMETRICAL SEMIANNUAL */
    t[3] = (p[15] + p[16] * plg[0][2]) * cd18;

    /*  ASYMMETRICAL ANNUAL */
    t[4] =  f1 * (p[9] * plg[0][1] + p[10] * plg[0][3]) * cd14;

    /*  ASYMMETRICAL SEMIANNUAL */
    t[5] =  p[37] * plg[0][1] * cd39;

    /* DIURNAL */
    if (flags->sw[7]) {
        double t71, t72;
        t71  = (p[11]*plg[1][2])*cd14*flags->swc[5];
        t72  = (p[12]*plg[1][2])*cd14*flags->swc[5];
        t[6] = f2 * ((p[3] * plg[1][1] + p[4] * plg[1][3] + p[27] * plg[1][5] + t71) * 
                     (*ctloc) + (p[6] * plg[1][1] + p[7] * plg[1][3] + p[28] * plg[1][5] 
                              + t72) * (*stloc));
    }

    /* SEMIDIURNAL */
    if (flags->sw[8]) {
        double t81, t82;
        t81  = (p[23] * plg[2][3] + p[35] * plg[2][5]) * cd14 * flags->swc[5];
        t82  = (p[33] * plg[2][3] + p[36] * plg[2][5]) * cd14 * flags->swc[5];
        t[7] = f2 *((p[5] * plg[2][2] + p[41] * plg[2][4] + t81) * (*c2tloc) +(p[8] * plg[2][2] + p[42] * plg[2][4] + t82) * (*s2tloc));
    }

    /* TERDIURNAL */
    if (flags->sw[14]) {
        t[13] = f2 * ((p[39] * plg[3][3] + (p[93] * plg[3][4] + p[46] * plg[3][6]) * cd14 * flags->swc[5]) * (*s3tloc) +(p[40] * plg[3][3] + (p[94] * plg[3][4] + p[48] * plg[3][6]) * cd14 * flags->swc[5]) * (*c3tloc));
    }

    /* magnetic activity based on daily ap */
    if (flags->sw[9] == -1) {
        ap = ap_a;
        if (p[51] != 0) {
            double exp1;
            exp1 = exp(-10800.0 * sqrt(p[51] * p[51]) / (1.0 + p[138] * (45.0 - sqrt(lat * lat))));
            if (exp1 > 0.99999){
                exp1 = 0.99999;
            }
            if (p[24] < 1.0E-4){
                p[24] = 1.0E-4;
            }
            apt[0] = sg0(exp1, p, ap);
            /* apt[1] = sg2(exp1, p, ap);
               apt[2] = sg0(exp2, p, ap);
               apt[3] = sg2(exp2, p, ap);
            */
            if (flags->sw[9]) {
                t[8] = apt[0] * (p[50] + p[96] * plg[0][2] + p[54] * plg[0][4] + 
                                 (p[125] * plg[0][1] + p[126] * plg[0][3] + p[127] * plg[0][5]) * cd14 * flags->swc[5] + 
                                 (p[128] * plg[1][1] + p[129] * plg[1][3] + p[130] * plg[1][5]) * flags->swc[7] * 
                                 cos(HR * (tloc - p[131])));
            }
        }
    } else {
        double p44, p45;
        apd = ap_in - 4.0;
        p44 = p[43];
        p45 = p[44];
        if (p44 < 0){
            p44 = 1.0E-5;
        }
        *apdf = apd + (p45 - 1.0) * (apd + (exp(-p44 * apd) - 1.0) / p44);
        if (flags->sw[9]) {
            t[8] = (*apdf) * (p[32] + p[45] * plg[0][2] + p[34] * plg[0][4] + 
                           (p[100] * plg[0][1] + p[101] * plg[0][3] + p[102] * plg[0][5]) * cd14 * flags->swc[5] +
                           (p[121] * plg[1][1] + p[122] * plg[1][3] + p[123] * plg[1][5]) * flags->swc[7] *
                           cos(HR * (tloc - p[124])));
        }
    }

    if ((flags->sw[10]) && (lon > -1000.0)) {

        /* longitudinal */
        if (flags->sw[11]) {
            t[10] = (1.0 + p[80] * (dfa) * flags->swc[1]) * 
                ((p[64] * plg[1][2] + p[65] * plg[1][4] + p[66] * plg[1][6]
                  + p[103] * plg[1][1] + p[104] * plg[1][3] + p[105] * plg[1][5]
                  + flags->swc[5] * (p[109] * plg[1][1] + p[110] * plg[1][3] + p[111] * plg[1][5]) * cd14)* 
                 cos(DGTR * lon) 
                 +(p[90] * plg[1][2] + p[91] * plg[1][4] + p[92] * plg[1][6]
                   +p[106] * plg[1][1] + p[107] * plg[1][3] + p[108] * plg[1][5]
                   +flags->swc[5] * (p[112] * plg[1][1] + p[113] * plg[1][3] + p[114] * plg[1][5]) * cd14) * 
                 sin(DGTR * lon));
        }

        /* ut and mixed ut, longitude */
        if (flags->sw[12]){
            t[11] = (1.0 + p[95] * plg[0][1]) * (1.0 + p[81] * (dfa) * flags->swc[1]) *
                (1.0 + p[119] * plg[0][1] * flags->swc[5] * cd14) * 
                ((p[68] * plg[0][1] + p[69] * plg[0][3] + p[70] * plg[0][5]) *
                 cos(SR * (sec - p[71])));
            t[11] += flags->swc[11] *
                (p[76] * plg[2][3] + p[77] * plg[2][5] + p[78] * plg[2][7]) *
                cos(SR * (sec - p[79]) + 2.0 * DGTR * lon) * (1.0 + p[137] * (dfa) * flags->swc[1]);
        }

        /* ut, longitude magnetic activity */
        if (flags->sw[13]) {
            if (flags->sw[9] == -1) {
                if (p[51]) {
                    t[12] = apt[0] * flags->swc[11] * (1. + p[132] * plg[0][1]) *
                        ((p[52] * plg[1][2] + p[98] * plg[1][4] + p[67] * plg[1][6]) *
                         cos(DGTR * (lon - p[97])))
                        + apt[0] * flags->swc[11] * flags->swc[5] *
                        (p[133] * plg[1][1] + p[134] * plg[1][3] + p[135] * plg[1][5]) *
                        cd14 * cos(DGTR * (lon - p[136])) 
                        + apt[0] * flags->swc[12] * 
                        (p[55] * plg[0][1] + p[56] * plg[0][3] + p[57] * plg[0][5]) *
                        cos(SR * (sec - p[58]));
                }
            } else {
                t[12] = (*apdf) * flags->swc[11] * (1.0 + p[120] * plg[0][1]) *
                    ((p[60] * plg[1][2] + p[61] * plg[1][4] + p[62] * plg[1][6]) *
                     cos(DGTR * (lon - p[63])))
                    + (*apdf) * flags->swc[11] * flags->swc[5] * 
                    (p[115] * plg[1][1] + p[116] * plg[1][3] + p[117] * plg[1][5]) * 
                    cd14 * cos(DGTR * (lon - p[118])) 
                    + (*apdf) * flags->swc[12] * 
                    (p[83] * plg[0][1] + p[84] * plg[0][3] + p[85] * plg[0][5]) * 
                    cos(SR * (sec - p[75]));
            }			
        }
    }

    /* params not used: 82, 89, 99, 139-149 */
    tinf = p[30];
    for (i = 0; i < 14; i++){
        tinf = tinf + fabs(flags->sw[i+1]) * t[i];
    }
    return tinf;
}



/* ------------------------------------------------------------------- */
/* ------------------------------- GLOB7S ---------------------------- */
/* ------------------------------------------------------------------- */

double glob7s(double *p, int doy, double lon,
			  struct nrlmsise_flags *flags, 
			  double ctloc, double stloc, double c2tloc, double s2tloc, 
			  double c3tloc, double s3tloc, double apdf, double apt) {
/*    VERSION OF GLOBE FOR LOWER ATMOSPHERE 10/26/99 
 */
    double pset = 2.0;
    double t[14];
    double tt;
    double cd32, cd18, cd14, cd39;
    double p32, p18, p14, p39;
    int i,j;

    /* confirm parameter set */
    if (p[99] == 0){
        p[99] = pset;
    }
    if (p[99]!=pset) {
        return -1;
    }
    for (j = 0; j < 14; j++){
        t[j] = 0.0;
    }

    cd32 = cos(DR * (doy - p[31]));
    cd18 = cos(2.0 * DR * (doy - p[17]));
    cd14 = cos(DR * (doy - p[13]));
    cd39 = cos(2.0 * DR * (doy - p[38]));
    p32  = p[31];
    p18  = p[17];
    p14  = p[13];
    p39  = p[38];

    /* F10.7 */
    t[0] = p[21] * dfa;

    /* time independent */
    t[1] = p[1] * plg[0][2] + p[2] * plg[0][4] + p[22] * plg[0][6] + p[26] * plg[0][1] + p[14] * plg[0][3] + p[59] * plg[0][5];

    /* SYMMETRICAL ANNUAL */
    t[2]=(p[18] + p[47] * plg[0][2] + p[29] * plg[0][4]) * cd32;

    /* SYMMETRICAL SEMIANNUAL */
    t[3]=(p[15] + p[16] * plg[0][2] + p[30] * plg[0][4]) * cd18;

    /* ASYMMETRICAL ANNUAL */
    t[4]=(p[9] * plg[0][1] + p[10] * plg[0][3] + p[20] * plg[0][5]) * cd14;

    /* ASYMMETRICAL SEMIANNUAL */
    t[5]=(p[37] * plg[0][1]) * cd39;

    /* DIURNAL */
    if (flags->sw[7]) {
        double t71, t72;
        t71  = p[11] * plg[1][2] * cd14 * flags->swc[5];
        t72  = p[12] * plg[1][2] * cd14 * flags->swc[5];
        t[6] = ((p[3] * plg[1][1] + p[4] * plg[1][3] + t71) * ctloc + (p[6] * plg[1][1] + p[7] * plg[1][3] + t72) * stloc) ;
    }

    /* SEMIDIURNAL */
    if (flags->sw[8]) {
        double t81, t82;
        t81  = (p[23] * plg[2][3] + p[35] * plg[2][5]) * cd14 * flags->swc[5];
        t82  = (p[33] * plg[2][3] + p[36] * plg[2][5]) * cd14 * flags->swc[5];
        t[7] = ((p[5] * plg[2][2] + p[41] * plg[2][4] + t81) * c2tloc + (p[8] * plg[2][2] + p[42] * plg[2][4] + t82) * s2tloc);
    }

    /* TERDIURNAL */
    if (flags->sw[14]) {
        t[13] = p[39] * plg[3][3] * s3tloc + p[40] * plg[3][3] * c3tloc;
    }

    /* MAGNETIC ACTIVITY */
    if (flags->sw[9]) {
        if (flags->sw[9] == 1){
            t[8] = apdf * (p[32] + p[45] * plg[0][2] * flags->swc[2]);
        }
        if (flags->sw[9] == -1)	{
            t[8] = (p[50] * apt + p[96] * plg[0][2] * apt * flags->swc[2]);
        }
    }

    /* LONGITUDINAL */
    if (!((flags->sw[10] == 0) || (flags->sw[11] == 0) || ( lon <= -1000.0))) {
        t[10] = (1.0 + plg[0][1] * (p[80] * flags->swc[5] * cos(DR * (doy - p[81]))
                                    + p[85] * flags->swc[6] * cos(2.0 * DR * (doy - p[86])))
                 + p[83] * flags->swc[3] * cos(DR * (doy - p[84]))
                 + p[87] * flags->swc[4] * cos(2.0 * DR * (doy - p[88])))
            * ((p[64] * plg[1][2] + p[65] * plg[1][4] + p[66] * plg[1][6]
                + p[74] * plg[1][1] + p[75] * plg[1][3] + p[76] * plg[1][5]
                   ) * cos(DGTR * lon)
               + (p[90] * plg[1][2] + p[91] * plg[1][4] + p[92] * plg[1][6]
                  + p[77] * plg[1][1] + p[78] * plg[1][3] + p[79] * plg[1][5]
                   ) * sin(DGTR * lon));
    }
    tt = 0;
    for (i = 0; i < 14; i++){
        tt += fabs((double)flags->sw[i+1]) * t[i];
    }
    return tt;
}



/* ------------------------------------------------------------------- */
/* ------------------------------- GTD7 ------------------------------ */
/* ------------------------------------------------------------------- */

void gtd7(const int *doy, const double *sec, const double *alt, double *lat, 
          double *lon, const double *tloc, const double *f107a, const double *f107, 
		  double *ap, const double *aph, int numPoints, struct nrlmsise_flags *flags,
          double *dens, double *T) {
    double xmm;
    int mn3 = 5;
    double zn3[5]={32.5,20.0,15.0,10.0,0.0};
    int mn2 = 4;
    double zn2[4]={72.5,55.0,45.0,32.5};
    double zmix=62.5;
    double dm28m;
    double tz;
    double dmc;
    double dmr;
    double dz28;
    static double dd;
    double dm28 = 0;
    static double meso_tn2[4];
    static double meso_tn3[5];
    static double meso_tgn2[2];
    static double meso_tgn3[2];
	static double gsurf;
	static double re;
	static double ctloc, stloc;
    static double c2tloc, s2tloc;
    static double s3tloc, c3tloc; 
	static double apdf;
    static double apt[4];
	static double meso_tn1[5];
	static double meso_tgn1[2];
	int i;

    tselec(flags);
    
    xmm = pdm[2][4];

    for (i = 0; i < numPoints; i++) {

	glatf( lat[i], &gsurf, &re, flags->sw[2]);

 	gts7( doy[i], sec[i], alt[i], lat[i], 
              lon[i], tloc[i], f107a[i], f107[i], ap[i], 
              aph, numPoints, i, flags, zn2[0], dens, T, 
			  &dm28, gsurf, re, &ctloc, &stloc, 
			  &c2tloc, &s2tloc, &c3tloc, &s3tloc, &apdf, &apt[0], &meso_tn1[0], &meso_tgn1[0] );

	/* metric adjustment */
	dm28m = dm28 * 1.0E6;


	if (alt[i] < zn2[0]) {

/*       LOWER MESOSPHERE/UPPER STRATOSPHERE (between zn3[0] and zn2[0])
 *         Temperature at nodes and gradients at end nodes
 *         Inverse temperature a linear function of spherical harmonics
 */
            meso_tgn2[0] = meso_tgn1[1];
            meso_tn2[0]  = meso_tn1[4];
            meso_tn2[1]  = pma[0][0] * pavgm[0] / (1.0 - flags->sw[20] * glob7s(pma[0], doy[i], 
                                                                                lon[i], flags,ctloc,stloc,c2tloc,s2tloc,c3tloc,s3tloc,apdf,apt[0]));
            meso_tn2[2]  = pma[1][0] * pavgm[1] / (1.0 - flags->sw[20] * glob7s(pma[1], doy[i], 
                                                                                lon[i], flags,ctloc,stloc,c2tloc,s2tloc,c3tloc,s3tloc,apdf,apt[0]));
            meso_tn2[3]  = pma[2][0] * pavgm[2] / (1.0 - flags->sw[20] * flags->sw[22] * glob7s(pma[2], doy[i], 
                                                                                                lon[i], flags,ctloc,stloc,c2tloc,s2tloc,c3tloc,s3tloc,apdf,apt[0]));
            meso_tgn2[1] = pavgm[8] * pma[9][0] * (1.0 + flags->sw[20] * flags->sw[22] * glob7s(pma[9], doy[i],
                                                                                                lon[i], flags,ctloc,stloc,c2tloc,s2tloc,c3tloc,s3tloc,apdf,apt[0])) * meso_tn2[3] * meso_tn2[3]/((pma[2][0] * pavgm[2])*(pma[2][0] * pavgm[2]));
            meso_tn3[0]  = meso_tn2[3];

            if (alt[i] < zn3[0]) {
/*       LOWER STRATOSPHERE AND TROPOSPHERE (below zn3[0])
 *         Temperature at nodes and gradients at end nodes
 *         Inverse temperature a linear function of spherical harmonics
 */
		meso_tgn3[0] = meso_tgn2[1];
		meso_tn3[1]  = pma[3][0] * pavgm[3] / (1.0 - flags->sw[22] * glob7s(pma[3], doy[i], 
                                                                                    lon[i], flags,ctloc,stloc,c2tloc,s2tloc,c3tloc,s3tloc,apdf,apt[0]));
		meso_tn3[2]  = pma[4][0] * pavgm[4] / (1.0 - flags->sw[22] * glob7s(pma[4], doy[i], 
                                                                                    lon[i], flags,ctloc,stloc,c2tloc,s2tloc,c3tloc,s3tloc,apdf,apt[0]));
		meso_tn3[3]  = pma[5][0] * pavgm[5] / (1.0 - flags->sw[22] * glob7s(pma[5], doy[i], 
                                                                                    lon[i], flags,ctloc,stloc,c2tloc,s2tloc,c3tloc,s3tloc,apdf,apt[0]));
		meso_tn3[4]  = pma[6][0] * pavgm[6] / (1.0 - flags->sw[22] * glob7s(pma[6], doy[i], 
                                                                                    lon[i], flags,ctloc,stloc,c2tloc,s2tloc,c3tloc,s3tloc,apdf,apt[0]));
		meso_tgn3[1] = pma[7][0] * pavgm[7] * (1.0 + flags->sw[22] * glob7s(pma[7], doy[i],
                                                                                    lon[i], flags,ctloc,stloc,c2tloc,s2tloc,c3tloc,s3tloc,apdf,apt[0])) * meso_tn3[4] * meso_tn3[4] / ((pma[6][0] * pavgm[6])*(pma[6][0] * pavgm[6]));
            }

            /* LINEAR TRANSITION TO FULL MIXING BELOW zn2[0] */

            dmc = 0;
            if (alt[i] > zmix){
		dmc = 1.0 - (zn2[0] - alt[i])/(zn2[0] - zmix);
            }
            dz28 = dens[i + numPoints * 2];
 	
            /**** N2 density ****/
            dmr = dens[i + numPoints * 2] / dm28m - 1.0;
            dens[i + numPoints * 2] = densm(alt[i], dm28m, xmm, &tz, mn3, zn3, meso_tn3, meso_tgn3, mn2, zn2, meso_tn2, meso_tgn2, gsurf, re);
            dens[i + numPoints * 2] = dens[i + numPoints * 2] * (1.0 + dmr * dmc);

            /**** HE density ****/
            dmr = dens[i] / (dz28 * pdm[0][1]) - 1.0;
            dens[i] = dens[i + numPoints * 2] * pdm[0][1] * (1.0 + dmr * dmc);

            /**** O density ****/
            dens[i + numPoints] = 0;
            dens[i + numPoints * 8] = 0;

            /**** O2 density ****/
            dmr = dens[i + numPoints * 3] / (dz28 * pdm[3][1]) - 1.0;
            dens[i + numPoints * 3] = dens[i + numPoints * 2] * pdm[3][1] * (1.0 + dmr * dmc);

            /**** AR density ***/
            dmr = dens[i + numPoints * 4] / (dz28 * pdm[4][1]) - 1.0;
            dens[i + numPoints * 4] = dens[i + numPoints * 2] * pdm[4][1] * (1.0 + dmr * dmc);

            /**** Hydrogen density ****/
            dens[i + numPoints * 6] = 0;

            /**** Atomic nitrogen density ****/
            dens[i + numPoints * 7] = 0;

            /**** Total mass density */
            dens[i + numPoints * 5] = 1.66E-24 * (4.0 * dens[i] + 16.0 * dens[i + numPoints] + 28.0 * dens[i + numPoints * 2] + 32.0 * dens[i + numPoints * 3] + 40.0 * dens[i + numPoints * 4] + dens[i + numPoints * 6] + 14.0 * dens[i + numPoints * 7]);

            /* adjust to metric */
            dens[i + numPoints * 5] = dens[i + numPoints * 5] / 1000;

            /**** temperature at altitude ****/
            dd = densm(alt[i], 1.0, 0, &tz, mn3, zn3, meso_tn3, meso_tgn3, mn2, zn2, meso_tn2, meso_tgn2, gsurf, re);
            T[i + numPoints] = tz;

	}
    }
    return;
}



/* ------------------------------------------------------------------- */
/* ------------------------------- GTD7D ----------------------------- */
/* ------------------------------------------------------------------- */

void gtd7d(const int *doy, const double *sec, const double *alt, double *lat, 
           double *lon, const double *lst, const double *f107a, 
		   const double *f107, double *ap, const double *aph_in, int numPoints,  
           struct nrlmsise_flags *flags, double *dens, double *T ) {

    int i;
    
    gtd7(doy, sec, alt, lat, lon, lst, f107a, f107, ap, 
         aph_in, numPoints, flags, dens, T);

    for (i = 0; i < numPoints; i++) {
	/**** Total mass density */
	dens[i + numPoints * 5] = 1.66E-24 * (4.0 * dens[i] + 16.0 * dens[i + numPoints] + 28.0 * dens[i + numPoints * 2] + 32.0 * dens[i + numPoints * 3] + 40.0 * dens[i + numPoints * 4] + dens[i + numPoints * 6] + 14.0 * dens[i + numPoints * 7] + 16.0 *  dens[i + numPoints * 8]);

        /* adjust to metric */
	dens[i + numPoints * 5] = dens[i + numPoints * 5] / 1000;
    }
}
 
/* ------------------------------------------------------------------- */
/* ------------------------------- GTS7 ------------------------------ */
/* ------------------------------------------------------------------- */

void gts7(const int doy, const double sec, const double alt, double lat, 
          double lon, const double tloc, const double f107a, const double f107, 
		  double ap, const double *aph, int numPoints, int i, 
          struct nrlmsise_flags *flags, double zn20, double *dens, double *T, 
	      double *dm28, double gsurf, double re, double *ctloc, double *stloc, 
		  double *c2tloc, double *s2tloc, double *c3tloc, double *s3tloc, 
		  double *apdf, double *apt, double *meso_tn1, 
		  double *meso_tgn1) {
/*     Thermospheric portion of NRLMSISE-00
 *     See GTD7 for more extensive comments
 *     alt > 72.5 km! 
 */
    double za;
    int j, k;
    double ddum, z;
    double zn1[5] = {120.0, 110.0, 100.0, 90.0, 72.5};
    double tinf;
    int mn1 = 5;
    double g0var;
    double tlb;
    double s, z0, t0, tr12;
    double db01, db04, db14, db16, db28, db32, db40, db48;
    double zh28, zh04, zh16, zh32, zh40, zh01, zh14;
    double zhm28, zhm04, zhm16, zhm32, zhm40, zhm01, zhm14;
    double xmd;
    double b28, b04, b16, b32, b40, b01, b14;
    static double dm04, dm16, dm32, dm40, dm01, dm14;
    double tz;
    double g28, g4, g16, g32, g40, g1, g14;
    double zhf, xmm;
    double zc04, zc16, zc32, zc40, zc01, zc14;
    double hc04, hc16, hc32, hc40, hc01, hc14;
    double hcc16, hcc32, hcc01, hcc14;
    double zcc16, zcc32, zcc01, zcc14;
    double rc16, rc32, rc01, rc14;
    double rl;
    double g16h, db16h, tho, zsht, zmho, zsho;
    double alpha[9]={-0.38, 0.0, 0.0, 0.0, 0.17, 0.0, -0.38, 0.0, 0.0};
    double altl[8]={200.0, 300.0, 160.0, 250.0, 240.0, 450.0, 320.0, 450.0};
    double dd;
    double hc216, hcc232;
    double altt;

    za = pdl[1][15];
    zn1[0] = za;


    if ( alt > zn20 ){
        altt = alt;
    }
    else {
        altt = zn20;
    }

    for (j = 0; j < 9; j++) {
        dens[i + numPoints * j] = 0;
    }

    /* TINF VARIATIONS NOT IMPORTANT BELOW ZA OR ZN1(1) */
    tinf = ptm[0] * pt[0];

    if ( altt > zn1[0]){
        tinf = tinf * (1.0 + flags->sw[16] * globe7(pt,doy,sec,lat, 
                                                    lon,tloc,f107a,f107,ap,aph,
													flags,ctloc,stloc,c2tloc,s2tloc,c3tloc,s3tloc,apdf,apt));
    }
    T[i] = tinf; 
	
    /*  GRADIENT VARIATIONS NOT IMPORTANT BELOW ZN1(5) */
    g0var = ptm[3] * ps[0];

    if (altt > zn1[4]){
        g0var = g0var * (1.0 + flags->sw[19] * globe7(ps,doy,sec,lat, 
                                                lon,tloc,f107a,f107,ap,aph,flags,ctloc,stloc,c2tloc,s2tloc,c3tloc,s3tloc,apdf,apt));
    }

    tlb = ptm[1] * (1.0 + flags->sw[17] * globe7(pd[3],doy,sec,lat, 
                                                 lon,tloc,f107a,f107,ap,aph,flags,ctloc,stloc,c2tloc,s2tloc,c3tloc,s3tloc,apdf,apt)) * pd[3][0];
    s = g0var / (tinf - tlb);
	
/*      Lower thermosphere temp variations not significant for
 *       density above 300 km */
    meso_tn1[1]  = ptm[6] * ptl[0][0];
    meso_tn1[2]  = ptm[2] * ptl[1][0];
    meso_tn1[3]  = ptm[7] * ptl[2][0];
    meso_tn1[4]  = ptm[4] * ptl[3][0];
    meso_tgn1[1] = ptm[8] * pma[8][0] * meso_tn1[4] * meso_tn1[4] / ((ptm[4] * ptl[3][0]) * (ptm[4] * ptl[3][0]));

    if (altt < 300.0) {
        meso_tn1[1]  = meso_tn1[1] / (1.0 - flags->sw[18] * glob7s(ptl[0],doy, 
                                                                   lon, flags,*ctloc,*stloc,*c2tloc,*s2tloc,*c3tloc,*s3tloc,*apdf,apt[0]));
        meso_tn1[2]  = meso_tn1[2] / (1.0 - flags->sw[18] * glob7s(ptl[1],doy, 
                                                                   lon, flags,*ctloc,*stloc,*c2tloc,*s2tloc,*c3tloc,*s3tloc,*apdf,apt[0]));
        meso_tn1[3]  = meso_tn1[3] / (1.0 - flags->sw[18] * glob7s(ptl[2],doy, 
                                                                   lon, flags,*ctloc,*stloc,*c2tloc,*s2tloc,*c3tloc,*s3tloc,*apdf,apt[0]));
        meso_tn1[4]  = meso_tn1[4] / (1.0 - flags->sw[18] * flags->sw[20] * glob7s(ptl[3],doy, 
                                                                                   lon, flags,*ctloc,*stloc,*c2tloc,*s2tloc,*c3tloc,*s3tloc,*apdf,apt[0]));
        meso_tgn1[1] = ptm[8] * pma[8][0] * meso_tn1[4] * meso_tn1[4] / ((ptm[4] * ptl[3][0]) * (ptm[4] * ptl[3][0]));
        meso_tgn1[1] = meso_tgn1[1] * (1.0 + flags->sw[18] * flags->sw[20] * glob7s(pma[8],doy, 
                                                                                    lon, flags,*ctloc,*stloc,*c2tloc,*s2tloc,*c3tloc,*s3tloc,*apdf,apt[0]));
    } 
	
    z0   = zn1[3];
    t0   = meso_tn1[3];
    tr12 = 1.0;

    /* N2 variation factor at Zlb */
    g28 = flags->sw[21] * globe7(pd[2],doy,sec,lat, 
                                 lon,tloc,f107a,f107,ap,aph, flags,ctloc,stloc,c2tloc,s2tloc,c3tloc,s3tloc,apdf,apt);
	
    /* VARIATION OF TURBOPAUSE HEIGHT */
    zhf = pdl[1][24] * (1.0 + flags->sw[5] * pdl[0][24] * sin(DGTR * lat) * cos(DR * ( doy - pt[13])));
    T[i] = tinf;
    xmm = pdm[2][4];
    z   = altt;

    /**** N2 DENSITY ****/

    /* Diffusive density at Zlb */
    db28 = pdm[2][0] * exp(g28) * pd[2][0];
    /* Diffusive density at Alt */
    dens[i + numPoints * 2] = densu(z, db28, tinf, tlb, 28.0, alpha[2], &T[i + numPoints], ptm[5], s, mn1, zn1, meso_tn1, meso_tgn1, gsurf, re);
    dd = dens[i + numPoints * 2];
    /* Turbopause */
    zh28  = pdm[2][2] * zhf;
    zhm28 = pdm[2][3] * pdl[1][5]; 
    xmd   = 28.0 - xmm;
    /* Mixed density at Zlb */
    b28 = densu(zh28, db28, tinf, tlb, xmd, (alpha[2] - 1.0), &tz, ptm[5], s, mn1, zn1, meso_tn1, meso_tgn1, gsurf, re);
    if ((flags->sw[15]) && (z <= altl[2])) {
        /*  Mixed density at Alt */
        *dm28 = densu(z, b28, tinf, tlb, xmm, alpha[2], &tz, ptm[5], s, mn1, zn1, meso_tn1, meso_tgn1, gsurf, re);
        /*  Net density at Alt */
        dens[i + numPoints * 2] = dnet(dens[i + numPoints * 2], *dm28, zhm28, xmm, 28.0);
    }

    /**** HE DENSITY ****/

    /*   Density variation factor at Zlb */
    g4 = flags->sw[21] * globe7(pd[0],doy,sec,lat, 
                                lon,tloc,f107a,f107,ap,aph, flags,ctloc,stloc,c2tloc,s2tloc,c3tloc,s3tloc,apdf,apt);
    /*  Diffusive density at Zlb */
    db04 = pdm[0][0] * exp(g4) * pd[0][0];
    /*  Diffusive density at Alt */
    dens[i] = densu(z, db04, tinf, tlb, 4., alpha[0], &T[i + numPoints], ptm[5], s, mn1, zn1, meso_tn1, meso_tgn1, gsurf, re);
    dd = dens[i];
    if ((flags->sw[15]) && (z < altl[0])) {
        /*  Turbopause */
        zh04 = pdm[0][2];
        /*  Mixed density at Zlb */
        b04 = densu(zh04, db04, tinf, tlb, 4. - xmm, alpha[0] - 1., &T[i + numPoints], ptm[5], s, mn1, zn1, meso_tn1, meso_tgn1, gsurf, re);
        /*  Mixed density at Alt */
        dm04 = densu(z, b04, tinf, tlb, xmm, 0., &T[i + numPoints], ptm[5], s, mn1, zn1, meso_tn1, meso_tgn1, gsurf, re);
        zhm04 = zhm28;
        /*  Net density at Alt */
        dens[i] = dnet(dens[i], dm04, zhm04, xmm, 4.);
        /*  Correction to specified mixing ratio at ground */
        rl = log( b28 * pdm[0][1] / b04);
        zc04 = pdm[0][4] * pdl[1][0];
        hc04 = pdm[0][5] * pdl[1][1];
        /*  Net density corrected at Alt */
        dens[i] = dens[i] * ccor(z, rl, hc04, zc04);
    }


    /**** O DENSITY ****/

    /*  Density variation factor at Zlb */
    g16 = flags->sw[21] * globe7(pd[1],doy,sec,lat, 
                                 lon,tloc,f107a,f107,ap,aph, flags,ctloc,stloc,c2tloc,s2tloc,c3tloc,s3tloc,apdf,apt);
    /*  Diffusive density at Zlb */
    db16 =  pdm[1][0] * exp(g16) * pd[1][0];
    /*   Diffusive density at Alt */
    dens[i + numPoints] = densu(z, db16, tinf, tlb, 16., alpha[1], &T[i + numPoints], ptm[5], s, mn1, zn1, meso_tn1, meso_tgn1, gsurf, re);
    dd = dens[i + numPoints];
    if ((flags->sw[15]) && (z <= altl[1])) {
        /*   Turbopause */
        zh16 = pdm[1][2];
        /*  Mixed density at Zlb */
        b16 = densu(zh16, db16, tinf, tlb, 16.0 - xmm, (alpha[1] - 1.0), &T[i + numPoints], ptm[5], s, mn1, zn1, meso_tn1, meso_tgn1, gsurf, re);
        /*  Mixed density at Alt */
        dm16 = densu(z, b16, tinf, tlb, xmm, 0., &T[i + numPoints], ptm[5], s, mn1, zn1, meso_tn1, meso_tgn1, gsurf, re);
        zhm16 = zhm28;
        /*  Net density at Alt */
        dens[i + numPoints] = dnet(dens[i + numPoints], dm16, zhm16, xmm, 16.);
        rl = pdm[1][1] * pdl[1][16] * (1.0 + flags->sw[1] * pdl[0][23] * (f107a - 150.0));
        hc16 = pdm[1][5] * pdl[1][3];
        zc16 = pdm[1][4] * pdl[1][2];
        hc216 = pdm[1][5] * pdl[1][4];
        dens[i + numPoints] = dens[i + numPoints] * ccor2(z, rl, hc16, zc16, hc216);
        /*   Chemistry correction */
        hcc16 = pdm[1][7] * pdl[1][13];
        zcc16 = pdm[1][6] * pdl[1][12];
        rc16 = pdm[1][3] * pdl[1][14];
        /*  Net density corrected at Alt */
        dens[i + numPoints] = dens[i + numPoints] * ccor(z, rc16, hcc16, zcc16);
    }

    /**** O2 DENSITY ****/

    /*   Density variation factor at Zlb */
    g32 = flags->sw[21] * globe7(pd[4],doy,sec,lat, 
                                 lon,tloc,f107a,f107,ap,aph, flags,ctloc,stloc,c2tloc,s2tloc,c3tloc,s3tloc,apdf,apt);
    /*  Diffusive density at Zlb */
    db32 = pdm[3][0] * exp(g32) * pd[4][0];
    /*   Diffusive density at Alt */
    dens[i + numPoints * 3] = densu(z, db32, tinf, tlb, 32., alpha[3], &T[i + numPoints], ptm[5], s, mn1, zn1, meso_tn1, meso_tgn1, gsurf, re);
    dd = dens[i + numPoints * 3];
    if (flags->sw[15]) {
        if (z <= altl[3]) {
            /*   Turbopause */
            zh32 = pdm[3][2];
            /*  Mixed density at Zlb */
            b32 = densu(zh32, db32, tinf, tlb, 32. - xmm, alpha[3] - 1., &T[i + numPoints], ptm[5], s, mn1, zn1, meso_tn1, meso_tgn1, gsurf, re);
            /*  Mixed density at Alt */
            dm32 = densu(z, b32, tinf, tlb, xmm, 0., &T[i + numPoints], ptm[5], s, mn1, zn1, meso_tn1, meso_tgn1, gsurf, re);
            zhm32 = zhm28;
            /*  Net density at Alt */
            dens[i + numPoints * 3] = dnet(dens[i + numPoints * 3], dm32, zhm32, xmm, 32.);
            /*   Correction to specified mixing ratio at ground */
            rl = log(b28 * pdm[3][1] / b32);
            hc32 = pdm[3][5] * pdl[1][7];
            zc32 = pdm[3][4] * pdl[1][6];
            dens[i + numPoints * 3] = dens[i + numPoints * 3] * ccor(z, rl, hc32, zc32);
        }
        /*  Correction for general departure from diffusive equilibrium above Zlb */
        hcc32 = pdm[3][7] * pdl[1][22];
        hcc232 = pdm[3][7] * pdl[0][22];
        zcc32 = pdm[3][6] * pdl[1][21];
        rc32 = pdm[3][3] * pdl[1][23] * (1. + flags->sw[1] * pdl[0][23] * (f107a - 150.));
        /*  Net density corrected at Alt */
        dens[i + numPoints * 3] = dens[i + numPoints * 3] * ccor2(z, rc32, hcc32, zcc32, hcc232);
    }

    /**** AR DENSITY ****/

    /*   Density variation factor at Zlb */
    g40 = flags->sw[20] * globe7(pd[5],doy,sec,lat, 
                                 lon,tloc,f107a,f107,ap,aph, flags,ctloc,stloc,c2tloc,s2tloc,c3tloc,s3tloc,apdf,apt);
    /*  Diffusive density at Zlb */
    db40 = pdm[4][0] * exp(g40) * pd[5][0];
    /*   Diffusive density at Alt */
    dens[i + numPoints * 4] = densu(z, db40, tinf, tlb, 40., alpha[4], &T[i + numPoints], ptm[5], s, mn1, zn1, meso_tn1, meso_tgn1, gsurf, re);
    dd = dens[i + numPoints * 4];
    if ((flags->sw[15]) && (z <= altl[4])) {
        /*   Turbopause */
        zh40 = pdm[4][2];
        /*  Mixed density at Zlb */
        b40 = densu(zh40, db40, tinf, tlb, 40. - xmm, alpha[4] - 1., &T[i + numPoints], ptm[5], s, mn1, zn1, meso_tn1, meso_tgn1, gsurf, re);
        /*  Mixed density at Alt */
        dm40 = densu(z, b40, tinf, tlb, xmm, 0., &T[i + numPoints], ptm[5], s, mn1, zn1, meso_tn1, meso_tgn1, gsurf, re);
        zhm40 = zhm28;
        /*  Net density at Alt */
        dens[i + numPoints * 4] = dnet(dens[i + numPoints * 4], dm40, zhm40, xmm, 40.);
        /*   Correction to specified mixing ratio at ground */
        rl = log(b28 * pdm[4][1] / b40);
        hc40 = pdm[4][5] * pdl[1][9];
        zc40 = pdm[4][4] * pdl[1][8];
        /*  Net density corrected at Alt */
        dens[i + numPoints * 4] = dens[i + numPoints * 4] * ccor(z, rl, hc40, zc40);
    }


    /**** HYDROGEN DENSITY ****/

    /*   Density variation factor at Zlb */
    g1 = flags->sw[21] * globe7(pd[6],doy,sec,lat, 
                                lon,tloc,f107a,f107,ap,aph, flags,ctloc,stloc,c2tloc,s2tloc,c3tloc,s3tloc,apdf,apt);
    /*  Diffusive density at Zlb */
    db01 = pdm[5][0] * exp(g1) * pd[6][0];
    /*   Diffusive density at Alt */
    dens[i + numPoints * 6] = densu(z, db01, tinf, tlb, 1., alpha[6], &T[i + numPoints], ptm[5], s, mn1, zn1, meso_tn1, meso_tgn1, gsurf, re);
    dd = dens[i + numPoints * 6];
    if ((flags->sw[15]) && (z <= altl[6])) {
        /*   Turbopause */
        zh01 = pdm[5][2];
        /*  Mixed density at Zlb */
        b01  = densu(zh01, db01, tinf, tlb, 1. - xmm, alpha[6] - 1., &T[i + numPoints], ptm[5], s, mn1, zn1, meso_tn1, meso_tgn1, gsurf, re);
        /*  Mixed density at Alt */
        dm01  = densu(z, b01, tinf, tlb, xmm, 0., &T[i + numPoints], ptm[5], s, mn1, zn1, meso_tn1, meso_tgn1, gsurf, re);
        zhm01 = zhm28;
        /*  Net density at Alt */
        dens[i + numPoints * 6] = dnet(dens[i + numPoints * 6], dm01, zhm01, xmm, 1.);
        /*   Correction to specified mixing ratio at ground */
        rl   = log(b28 * pdm[5][1] * sqrt(pdl[1][17] * pdl[1][17]) / b01);
        hc01 = pdm[5][5] * pdl[1][11];
        zc01 = pdm[5][4] * pdl[1][10];
        dens[i + numPoints * 6] = dens[i + numPoints * 6] * ccor(z, rl, hc01, zc01);
        /*   Chemistry correction */
        hcc01 = pdm[5][7] * pdl[1][19];
        zcc01 = pdm[5][6] * pdl[1][18];
        rc01  = pdm[5][3] * pdl[1][20];
        /*  Net density corrected at Alt */
        dens[i + numPoints * 6] = dens[i + numPoints * 6] * ccor(z, rc01, hcc01, zcc01);
    }
    /**** ATOMIC NITROGEN DENSITY ****/

    /*   Density variation factor at Zlb */
    g14 = flags->sw[21] * globe7(pd[7],doy,sec,lat, 
                                 lon,tloc,f107a,f107,ap,aph, flags,ctloc,stloc,c2tloc,s2tloc,c3tloc,s3tloc,apdf,apt);
    /*  Diffusive density at Zlb */
    db14 = pdm[6][0] * exp(g14)*pd[7][0];
    /*   Diffusive density at Alt */
    dens[i + numPoints * 7] = densu(z, db14, tinf, tlb, 14., alpha[7], &T[i + numPoints], ptm[5], s, mn1, zn1, meso_tn1, meso_tgn1, gsurf, re);
    dd = dens[i + numPoints * 7];
    if ((flags->sw[15]) && (z <= altl[7])) {
        /*   Turbopause */
        zh14  = pdm[6][2];
        /*  Mixed density at Zlb */
        b14   = densu(zh14, db14, tinf, tlb, 14. - xmm, alpha[7] - 1., &T[i + numPoints], ptm[5], s, mn1, zn1, meso_tn1, meso_tgn1, gsurf, re);
        /*  Mixed density at Alt */
        dm14  = densu(z, b14, tinf, tlb, xmm, 0., &T[i + numPoints], ptm[5], s, mn1, zn1, meso_tn1, meso_tgn1, gsurf, re);
        zhm14 = zhm28;
        /*  Net density at Alt */
        dens[i + numPoints * 7] = dnet(dens[i + numPoints * 7], dm14, zhm14, xmm, 14.);
        /*   Correction to specified mixing ratio at ground */
        rl   = log(b28 * pdm[6][1] * sqrt(pdl[0][2] * pdl[0][2]) / b14);
        hc14 = pdm[6][5] * pdl[0][1];
        zc14 = pdm[6][4] * pdl[0][0];
        dens[i + numPoints * 7] = dens[i + numPoints * 7] * ccor(z, rl, hc14, zc14);
        /*   Chemistry correction */
        hcc14 = pdm[6][7] * pdl[0][4];
        zcc14 = pdm[6][6] * pdl[0][3];
        rc14  = pdm[6][3] * pdl[0][5];
        /*  Net density corrected at Alt */
        dens[i + numPoints * 7] = dens[i + numPoints * 7] * ccor(z, rc14, hcc14, zcc14);
    }


    /**** Anomalous OXYGEN DENSITY ****/

    g16h  = flags->sw[21] * globe7(pd[8],doy,sec,lat, 
                                   lon,tloc,f107a,f107,ap,aph, flags,ctloc,stloc,c2tloc,s2tloc,c3tloc,s3tloc,apdf,apt);
    db16h = pdm[7][0] * exp(g16h) * pd[8][0];
    tho   = pdm[7][9] * pdl[0][6];
    dd    = densu(z, db16h, tho, tho, 16., alpha[8], &T[i + numPoints], ptm[5], s, mn1, zn1, meso_tn1, meso_tgn1, gsurf, re);
    zsht  = pdm[7][5];
    zmho  = pdm[7][4];
    zsho  = scalh(zmho, 16.0, tho, gsurf, re);
    dens[i + numPoints * 8] = dd * exp(-zsht / zsho * (exp(-(z - zmho) / zsht) - 1.));


    /* total mass density */
    dens[i + numPoints * 5] = 1.66E-24 * (4.0 * dens[i] + 16.0 * dens[i + numPoints] + 28.0 * dens[i + numPoints * 2] + 32.0 * dens[i + numPoints * 3] + 40.0 * dens[i + numPoints * 4] + dens[i + numPoints * 6] + 14.0 * dens[i + numPoints * 7]);
    db48 = 1.66E-24 * (4.0 * db04 + 16.0 * db16 + 28.0 * db28 + 32.0 * db32 + 40.0 * db40 + db01 + 14.0 * db14);



    /* temperature */
    z = sqrt(altt * altt);
    ddum = densu(z, 1.0, tinf, tlb, 0.0, 0.0, &T[i + numPoints], ptm[5], s, mn1, zn1, meso_tn1, meso_tgn1, gsurf, re);

    /* output in kilograms and meters */
    for(k = 0; k < 9; k++){
        dens[i + numPoints * k] = dens[i + numPoints * k] * 1.0E6;
    }
    dens[i + numPoints * 5] = dens[i + numPoints * 5] / 1000;

}
