/**
 * WidgetViewManager is responsible for handling the views for an Web Figure and
 * all Component views under Web Figure
 *
 * The WidgetViewManager requires:
 * - A WidgetData for an Web Figure
 * - a targetLocation where the Web Figure should render its content
 *
 * When the WidgetViewManager is created, it will:
 * - Render the Web figure to the border container of the UIContainer
 *
 * Copyright 2014 MathWorks, Inc.
 */

define([
    "dojo/_base/declare",
    "dojo/_base/lang",
    "dojo/on",
    "dojo/dom-style",
    "gbt/Locator"
],  function (declare, lang, on, domStyle, Locator) {

    return declare([], {

        /**
        * Creates a WidgetViewManager
        * @param figureWidgetData      - gbt/model/WidgetData                      WidgetData for Figure
        * @param targetLocation        - borderContainer from UiContainer          Where the Web Figure will be placed
        */
        constructor: function (figureWidgetData, targetLocation) {
            // Assertions
            if (!figureWidgetData) {
                throw "figureWidgetData must not be null / undefined";
            }

            if (!targetLocation) {
                throw "targetLocation must not be null / undefined";
            }

            // create WidgetView
            this._createWidgetView(figureWidgetData, targetLocation);

            //listen for future children under the Web Figure
            this._addChildListenersToWidgetData(figureWidgetData)
        },

         /**
         * ------------------------------------------------------------------------------------------------------
         * Private Functions
         * ------------------------------------------------------------------------------------------------------
         */

         /**
         * Helper function which attaches listeners to the WidgetData
         */
        _addChildListenersToWidgetData: function (widgetData) {
            //Add children  to the parent
            widgetData.on("widgetDataAdded", lang.hitch(this, function (event) {

                var newChildWidgetData = event.WidgetData;

                //get the Target node to place the child widget view
                var parentWidgetView = widgetData.getTargetNode();
                var parentDOMNode = parentWidgetView.getContentAreaDOMNode();

                //create the child component
                this._createWidgetView(newChildWidgetData, parentDOMNode);

                // recursively listen to future add / removes from this container
                this._addChildListenersToWidgetData(newChildWidgetData);
            }));

             //Remove children from the parent
            widgetData.on("widgetDataRemoved", lang.hitch(this, function (event) {
                var delChildWidgetData = event.WidgetData;

                //get the widgetView associated to teh widgetData
                var widgetView  =   delChildWidgetData.getTargetNode();

                if (widgetView) {
                    //destroy the WidgetView
                    widgetView.destroy();
                } else {
                    throw "widgetView is undefined for the widgetData";
                }
            }));
        },

        /**
        * Creates a WidgetView for the given WidgetData under the parent DOM
        * @param widgetData
        * @param target
        * @private
        */
        _createWidgetView : function (widgetData, target) {
            var type = widgetData.getType();
            var locator = new Locator();
            var component = locator.locate(type);
            var widgetView = new component(widgetData, target, ("qe-" + widgetData.getId()));

            if (type === "ContainerRoot") {
                //add the figure dijit to the border container
                var figure =  widgetView.getDijitHandle();
                //add the Figure to the border Container
                target.addChild(figure);
                //set the Default size for Canvas based on the size of the Figure containerNode
                figure.setDefaultDimensions();
            } else {
                //place the Component widget at parent widget containerNode
                domStyle.set(widgetView.getVCWidgetDOMNode(), "z-index", "1");
            }
                // set the target WidgetView as place holder for the children
            widgetData.setTargetNode(widgetView);
        }

    });
});

