/*
 * File: aerointerpolate.c
 *
 * Abstract:
 *    Interpolation helper functions 
 *
 *     
 * Copyright 1990-2013 The MathWorks, Inc.
 *
 * Author:  S. Gage      8-MAR-2006
 */

#include <math.h>
#include "export/include/aeroatmos/aerointerpolate.hpp"

/* Function: LookBinXd =====================================================
 *    Integrated index search function with extrapolation  
 *    or binary search algorithm to find index and interval 
 *    fraction.
 */

unsigned int LookBinXd(const double   altitude,
                       double        *fraction,
                       const double  *altitudeTab,
                       const unsigned int      maxIndex)
{
    unsigned int bottom = 0U;
    unsigned int idx;

    if (altitude < altitudeTab[bottom]) {
        idx = 0U;
    } 
    else if (altitude < altitudeTab[maxIndex - 1]) {
        unsigned int top = maxIndex;
        unsigned int notFound = 1U;
        unsigned int cnt;

        idx = bottom;
        for (cnt = maxIndex;(cnt > 0U) && (notFound == 1U);cnt = cnt >> 1U ) 
        {
            if (altitude < altitudeTab[idx]) {
                top = idx - 1U;
                idx = (bottom + top) >> 1U;

            } else if (altitude < altitudeTab[idx+1]) {
                notFound = 0U; /* index found break loop */
            } else {
                bottom = idx + 1U;
                idx = (bottom + top) >> 1U;
            }
        } 
    } 
    else {
        idx = maxIndex - 1U;
    }

    *fraction = (altitude        - altitudeTab[idx]) /
        (altitudeTab[idx+1] - altitudeTab[idx]);   
                        
    return(idx);
}

/* Function: LinearInterp ==================================================
 * Abstract:
 *   linearly interpolate.
 *
 */
static double LinearInterp(const double  *Tab,
                           const double fraction,
                           const unsigned int idx)
{
    double result    = Tab[idx] + fraction * ( Tab[idx+1] - Tab[idx] );

    return(result);
}

/* Function: LogInterp =====================================================
 * Abstract:
 *   logarithmically interpolate.
 *
 */
static double LogInterp(const double  *Tab,
                        const double fraction,
                        const unsigned int idx)
{
    double result    = Tab[idx] * pow( Tab[idx+1] / Tab[idx], fraction );

    return(result);
}

/* Function: InterpolateTempAndDens ======================================
 * Abstract:
 *   linearly interpolate temperature and logarithmically 
 *     interpolate density.
 *
 */
void InterpolateTempAndDens(const double altitude,
                            const double  *temperatureTab,
                            const double  *densityTab,
                            double  *temp, 
                            double  *density, 
                            const double  fraction,
                            const unsigned int     idx,
                            const double  lower_alt,
                            const double  upper_alt )

{           
    if ((altitude > lower_alt) || (altitude < upper_alt))
    { 
        /* Linear interpolation on temperature */
        *temp = LinearInterp( temperatureTab, fraction, idx);

        /* Logarithmic interpolation on density */
        *density = LogInterp( densityTab, fraction, idx);

    } else {
        /* outside defined altitude range hold values */
        *temp = temperatureTab[idx];
        *density = densityTab[idx];
    }
}
/* Function: AltitudeLimitLinearInterpolate  ===============================
 * Abstract:
 *   linearly interpolate with altitude limits
 *
 */
void AltitudeLimitLinearInterpolate(const double altitude,
                                    const double *Tab,
                                    const double  fraction,
                                    double *result,
                                    const unsigned int idx,
                                    const double lower_alt,
                                    const double upper_alt)
{
    if ((altitude > lower_alt) || (altitude < upper_alt))
    { 
        /* Linear interpolation  */
        *result = LinearInterp( Tab, fraction, idx);

    } else {
        /* outside defined altitude range hold values */
        *result = Tab[idx];
    }
}
/* Function: AltitudeLimitLogInterpolate  ==================================
 * Abstract:
 *   logarithmically interpolate with altitude limits
 *
 */
void AltitudeLimitLogInterpolate(const double altitude,
                                 const double *Tab,
                                 const double  fraction,
                                 double *result,
                                 const unsigned int  idx,
                                 const double lower_alt,
                                 const double upper_alt)
{
    if ((altitude > lower_alt) || (altitude < upper_alt))
    { 
        /* Logarithmic interpolation on density  */
        *result = LogInterp( Tab, fraction, idx);
    } else {
        /* outside defined altitude range hold values */
        *result = Tab[idx];
    }
}
