/* Copyright 2011 The MathWorks, Inc. */

/* 
 * File: ide_rtiostream.c
 *
 * RtIOStream over an IDE debugger link
 */

#include "rtiostream.h"
#include "xil_common.h"

/* RX_BUFFER_MEMUNIT_SIZE must be representable in size_t */
#define RX_BUFFER_MEMUNIT_SIZE (RTIOSTREAM_RX_BUFFER_BYTE_SIZE / MEM_UNIT_BYTES)

/* 
 * The following data is global so ide's have visibility
 */

/* buffer for data received from host */
volatile MemUnit_T IdeRtIOStreamRecvBuffer[RX_BUFFER_MEMUNIT_SIZE]; 
/* pointer to data to send to host */
volatile const MemUnit_T * IdeRtIOStreamSendDataPtr; 
/* size of data to send to host */
volatile size_t IdeRtIOStreamSendDataAvail;
/* define sizes that may be required by certain IDE implementations */
volatile MemUnit_T IdeRtIOStreamPtrSize = sizeof(void *) * MEM_UNIT_BYTES;
volatile MemUnit_T IdeRtIOStreamSizeTSize = sizeof(size_t) * MEM_UNIT_BYTES;

/* global function */
void pilDataBreakpoint(void);

/* local state */
static volatile MemUnit_T * recvDataPtr;
static volatile size_t recvDataAvail = 0;

void pilDataBreakpoint() {
    /* always reset recv info */
    recvDataPtr = &IdeRtIOStreamRecvBuffer[0];
    /* assume host transfers data in maximal chunks */
    recvDataAvail = RX_BUFFER_MEMUNIT_SIZE;
}

int rtIOStreamOpen(
      int    argc,
      void * argv[]) {
   UNUSED_PARAMETER(argc);
   UNUSED_PARAMETER(argv);
   /* workaround for certain compilers that can optimize away these
    * volatile variables unless they are accessed within this module */
   IdeRtIOStreamPtrSize = sizeof(void *) * MEM_UNIT_BYTES;
   IdeRtIOStreamSizeTSize = sizeof(size_t) * MEM_UNIT_BYTES;
   /* return a valid station id */
   return 0;
}

int rtIOStreamSend(
      int          streamID,
      const void * src, 
      size_t       size,
      size_t     * sizeSent) {
   /* outgoing data has already been buffered; transmit immediately */
   IdeRtIOStreamSendDataPtr = (MemUnit_T *) src;
   IdeRtIOStreamSendDataAvail = size;
   *sizeSent = size;
   /* breakpoint for host to read from buffer */
   pilDataBreakpoint();
   UNUSED_PARAMETER(streamID);
   return RTIOSTREAM_NO_ERROR;
}

int rtIOStreamRecv(
      int      streamID,
      void   * dst, 
      size_t   size,
      size_t * sizeRecvd) {   
   size_t transferAmount;   
   MemUnit_T * dstMemUnit = (MemUnit_T *) dst;
   *sizeRecvd = size;
   /* block until all data is received */
   while (size > 0) {
      if (recvDataAvail == 0) {
         /* breakpoint for host to write data to buffer */
         pilDataBreakpoint();         
      }
      transferAmount = (size_t) MIN(recvDataAvail, size);
      memcpy(dstMemUnit, (void *) recvDataPtr, transferAmount);
      size -= transferAmount;
      recvDataAvail -= transferAmount;
      dstMemUnit += transferAmount;
      recvDataPtr += transferAmount;
   }   
   UNUSED_PARAMETER(streamID);
   return RTIOSTREAM_NO_ERROR;
}

int rtIOStreamClose(
      int streamID) {
   /* do nothing */
   UNUSED_PARAMETER(streamID);
   return RTIOSTREAM_NO_ERROR;
}
