/* Copyright 2013-2014 The MathWorks, Inc. */

/*
 * File: xilcomms_rtiostream.c
 *
 */

/* include services headers */
#include "comms_interface.h"
#include "xilservice.h"
#ifdef CODE_INSTRUMENTATION_ENABLED
#include "codeinstrservice.h"
#endif
/* include rtIOStream interface to use */
#include "rtiostream.h"
/* include Target-side rtIOStream utility APIs */
#include "rtiostream_utils.h"
/* rx/tx buffer sizes info */
#include "rx_tx_buffer_sizes.h"
/* need UNUSED_PARAMETER */
#include "xil_common.h"

/* call xilProcessMsg */
extern void xilProcessMsg(const MemUnit_T * const src, uint16_T size);

/* store stream handle */
static int streamID;

#define APPLICATION_ID_SIZE (sizeof(MemUnit_T))
#define APPLICATION_ID_IDX 0
#define PAYLOAD_SIZE_IDX (APPLICATION_ID_IDX + APPLICATION_ID_SIZE)
#define PAYLOAD_IDX (PAYLOAD_SIZE_IDX + WRITE_DATA_AVAIL_SIZE)

/* allocate the buffers */
static MemUnit_T xilCommsBuffer[COMMSERVICE_RX_BUFFER_MEMUNIT_SIZE];
static IOUnit_T xilWriteBuffer[XIL_TX_BUFFER_MEMUNIT_SIZE];
#ifdef CODE_INSTRUMENTATION_ENABLED
static IOUnit_T codeInstrWriteBuffer[CODE_INSTR_TX_BUFFER_MEMUNIT_SIZE];
#endif

/* Returns pointer to corresponding buffer for application 
 * given its id. Returns NULL for invalid application ids.
 */
static IOUnit_T* getApplicationBufferPtr(
        const uint8_T appId){
        	
    switch(appId) {
        case XIL_SERVICE_ID:
                return &xilWriteBuffer[APPLICATION_ID_IDX];
#ifdef CODE_INSTRUMENTATION_ENABLED                            
        case CODEINSTR_SERVICE_ID:
                return &codeInstrWriteBuffer[APPLICATION_ID_IDX];
#endif            
        default:			
            return NULL;
    }    
}

/* provide access to XIL send buffer */
XILSERVICE_API_EXTERN_C uint8_T commsAllocXILBuffer(
        void** ppBuf,
        const uint16_T dataSize){
        
    /* buffer is already statically allocated */
    *ppBuf = NULL;
    
    UNUSED_PARAMETER(dataSize);    
        
    /* nothing to do as buffer is already statically allocated */
    return COMMS_SUCCESS;
}

/* return pointer to XIL buffer data */
XILSERVICE_API_EXTERN_C IOUnit_T * commsGetXILBufferDataPtr(
        void* const pBuf){
    
    /* buffer is already statically allocated */
    UNUSED_PARAMETER(pBuf);   
    return &xilWriteBuffer[PAYLOAD_IDX];
}

#ifdef CODE_INSTRUMENTATION_ENABLED
/* provide access to code instrumentation send buffer */
uint8_T commsAllocCodeInstrBuffer(
        void** ppBuf,
        const uint16_T dataSize){
        
    /* buffer is already statically allocated */
    *ppBuf = NULL;
    
    UNUSED_PARAMETER(dataSize);
        
    /* nothing to do as buffer is already statically allocated */
    return COMMS_SUCCESS;
}

/* return pointer to code instrumentation buffer data */
XILSERVICE_API_EXTERN_C IOUnit_T * commsGetCodeInstrBufferDataPtr(
        void* const pBuf){
    
    /* buffer is already statically allocated */
    UNUSED_PARAMETER(pBuf);    
    return &codeInstrWriteBuffer[PAYLOAD_IDX];
}
#endif

/* open rtiostream */
uint8_T commsCreate(
        const int argc, 
        void * argv[]){
    
    streamID = rtIOStreamOpen(argc, argv);
    if (streamID == RTIOSTREAM_ERROR) {
        return COMMS_ERROR;
    }
    return COMMS_SUCCESS;
}

/* close rtiostream */
uint8_T commsDestroy(void) {    
    return(rtIOStreamClose(streamID) == RTIOSTREAM_NO_ERROR);
}

/* call rtIOStreamBlockingSend to send data*/
XILSERVICE_API_EXTERN_C uint8_T commsEnqueueBuffer(        
        void* const pBuf,
        const uint8_T appId,
        const uint16_T dataSize) {
    
    int rtIOStreamErrorStatus;    
    MemUnit_T appIdMemUnit = (MemUnit_T)appId;
    uint32_T writeDataAvail = (uint32_T)dataSize;   
	uint32_T transferSize = writeDataAvail + BUFFER_HEADER_SIZE;           

    /* write size and application id before sending the buffer */
#ifdef HOST_WORD_ADDRESSABLE_TESTING
    /* writeDataAvail is in terms of IOUnit_T (uint8_T) - convert to MemUnit_T's
     * assume that writeDataAvail divides exactly */
    uint32_T writeDataAvailMemUnits = writeDataAvail / MEM_UNIT_BYTES;
#else
    uint32_T writeDataAvailMemUnits = writeDataAvail;
#endif
    
    /* get the pointer for the application buffer of the given app id */
    IOUnit_T* appBuffer = getApplicationBufferPtr(appId);
	if (appBuffer == NULL) {
		/* early return */
		return COMMS_ERROR;
	}
        
    UNUSED_PARAMETER(pBuf);
    
    memcpy((void *) &appBuffer[APPLICATION_ID_IDX], &appIdMemUnit, sizeof(appIdMemUnit));        
    memcpy((void *) &appBuffer[PAYLOAD_SIZE_IDX], &writeDataAvailMemUnits, sizeof(writeDataAvailMemUnits));    
    
   /* Blocks until all requested outgoing data is sent */
   rtIOStreamErrorStatus = rtIOStreamBlockingSend(streamID,
                                                 (const void *) appBuffer,
                                                 transferSize);
                                                      
   if (rtIOStreamErrorStatus == RTIOSTREAM_ERROR)
         return COMMS_ERROR;
   
   return COMMS_SUCCESS;
}

#define RECV_BUFFER_DATA_IDX 0

uint8_T commsRun(void) {
    
    int rtIOStreamErrorStatus;    
    uint32_T dataSize;
    MemUnit_T appId;    

    /* receive app id */
    rtIOStreamErrorStatus = rtIOStreamBlockingRecv(streamID,
                                                  (void *) &appId,
                                                   sizeof(appId));

    if (rtIOStreamErrorStatus == RTIOSTREAM_ERROR)
		return COMMS_ERROR;
    
    /* receive size of data contained in the buffer */
    rtIOStreamErrorStatus = rtIOStreamBlockingRecv(streamID,
                                                  (void *) &dataSize,
                                                   sizeof(dataSize));

    if (rtIOStreamErrorStatus == RTIOSTREAM_ERROR)
		return COMMS_ERROR;
    
#ifdef HOST_WORD_ADDRESSABLE_TESTING
	/* dataSize is in terms of MemUnit_T - convert to IOUnit_T (uint8_T). Note
	 * that dataSize will not overflow since the host already divided dataSize 
	 * by memUnit_T before transmission, so this multiplication should be safe
	 * and won't overflow.
	 */
	dataSize *= MEM_UNIT_BYTES;
#endif

    /* receive the data */
    rtIOStreamErrorStatus = rtIOStreamBlockingRecv(streamID,
                                                  (void *) &xilCommsBuffer[RECV_BUFFER_DATA_IDX],
                                                   dataSize);
    
    if (rtIOStreamErrorStatus == RTIOSTREAM_ERROR)
		return COMMS_ERROR;
    
    /* dispatch buffer to owning application */
    switch(appId) {
        case XIL_SERVICE_ID:
                xilProcessMsg(&(xilCommsBuffer[RECV_BUFFER_DATA_IDX]), (uint16_T)dataSize);
            break;
        default:
			return COMMS_ERROR;            
    }
	return COMMS_SUCCESS;      
}

